<?php
/**
 * Modelo de Usuário
 */

class User {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Criar novo usuário
     */
    public function create($data) {
        $sql = "INSERT INTO users (
            email, password_hash, first_name, last_name, 
            auth_provider, provider_id, currency, language
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $passwordHash = isset($data['password']) ? password_hash($data['password'], PASSWORD_BCRYPT) : null;
        
        $stmt->bind_param(
            'ssssssss',
            $data['email'],
            $passwordHash,
            $data['first_name'] ?? null,
            $data['last_name'] ?? null,
            $data['auth_provider'] ?? 'email',
            $data['provider_id'] ?? null,
            $data['currency'] ?? 'BRL',
            $data['language'] ?? 'pt-BR'
        );

        if (!$stmt->execute()) {
            throw new Exception("Erro ao criar usuário: " . $stmt->error);
        }

        return $this->db->insert_id;
    }

    /**
     * Obter usuário por ID
     */
    public function getById($id) {
        $sql = "SELECT * FROM users WHERE id = ? AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $id);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }

    /**
     * Obter usuário por email
     */
    public function getByEmail($email) {
        $sql = "SELECT * FROM users WHERE email = ? AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('s', $email);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }

    /**
     * Obter usuário por provider
     */
    public function getByProvider($provider, $providerId) {
        $sql = "SELECT * FROM users WHERE auth_provider = ? AND provider_id = ? AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ss', $provider, $providerId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }

    /**
     * Verificar senha
     */
    public function verifyPassword($plainPassword, $hash) {
        return password_verify($plainPassword, $hash);
    }

    /**
     * Atualizar usuário
     */
    public function update($id, $data) {
        $allowedFields = [
            'first_name', 'last_name', 'profile_picture', 'phone',
            'country', 'currency', 'language', 'preferences'
        ];

        $updates = [];
        $params = [];
        $types = '';

        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updates[] = "$field = ?";
                $params[] = $data[$field];
                $types .= 's';
            }
        }

        if (empty($updates)) {
            return false;
        }

        $params[] = $id;
        $types .= 'i';

        $sql = "UPDATE users SET " . implode(', ', $updates) . " WHERE id = ?";
        $stmt = $this->db->prepare($sql);

        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        
        if (!$stmt->execute()) {
            throw new Exception("Erro ao atualizar usuário: " . $stmt->error);
        }

        return true;
    }

    /**
     * Atualizar última data de login
     */
    public function updateLastLogin($id) {
        $sql = "UPDATE users SET last_login = NOW() WHERE id = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $id);
        
        return $stmt->execute();
    }

    /**
     * Verificar email
     */
    public function verifyEmail($id) {
        $sql = "UPDATE users SET email_verified = TRUE, email_verified_at = NOW() WHERE id = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $id);
        
        return $stmt->execute();
    }

    /**
     * Incrementar tentativas de login
     */
    public function incrementLoginAttempts($email) {
        $sql = "UPDATE users SET login_attempts = login_attempts + 1 WHERE email = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('s', $email);
        
        return $stmt->execute();
    }

    /**
     * Bloquear usuário
     */
    public function lockUser($email, $lockoutTime = 900) {
        $lockUntil = date('Y-m-d H:i:s', time() + $lockoutTime);
        $sql = "UPDATE users SET login_attempts = 0, locked_until = ? WHERE email = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ss', $lockUntil, $email);
        
        return $stmt->execute();
    }

    /**
     * Desbloquear usuário
     */
    public function unlockUser($email) {
        $sql = "UPDATE users SET login_attempts = 0, locked_until = NULL WHERE email = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('s', $email);
        
        return $stmt->execute();
    }

    /**
     * Verificar se usuário está bloqueado
     */
    public function isLocked($email) {
        $sql = "SELECT locked_until FROM users WHERE email = ? AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('s', $email);
        $stmt->execute();
        
        $result = $stmt->get_result()->fetch_assoc();
        
        if (!$result || !$result['locked_until']) {
            return false;
        }

        $lockUntil = strtotime($result['locked_until']);
        
        if (time() > $lockUntil) {
            $this->unlockUser($email);
            return false;
        }

        return true;
    }

    /**
     * Deletar usuário (soft delete)
     */
    public function delete($id) {
        $sql = "UPDATE users SET deleted_at = NOW() WHERE id = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $id);
        
        return $stmt->execute();
    }
}
?>
