<?php
/**
 * Modelo de Transação
 */

class Transaction {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Criar nova transação
     */
    public function create($userId, $data) {
        $sql = "INSERT INTO transactions (
            user_id, account_id, category_id, type, description, amount,
            currency, transaction_date, due_date, status, payment_method, notes
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param(
            'iissdsssss',
            $userId,
            $data['account_id'],
            $data['category_id'] ?? null,
            $data['type'],
            $data['description'] ?? '',
            $data['amount'],
            $data['currency'] ?? 'BRL',
            $data['transaction_date'] ?? date('Y-m-d'),
            $data['due_date'] ?? null,
            $data['status'] ?? 'completed',
            $data['payment_method'] ?? 'cash',
            $data['notes'] ?? null
        );

        if (!$stmt->execute()) {
            throw new Exception("Erro ao criar transação: " . $stmt->error);
        }

        return $this->db->insert_id;
    }

    /**
     * Obter transação por ID
     */
    public function getById($id, $userId) {
        $sql = "SELECT t.*, c.name as category_name, a.name as account_name
                FROM transactions t
                LEFT JOIN categories c ON t.category_id = c.id
                LEFT JOIN accounts a ON t.account_id = a.id
                WHERE t.id = ? AND t.user_id = ? AND t.deleted_at IS NULL";
        
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $id, $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }

    /**
     * Obter transações do usuário com filtros
     */
    public function getByUserId($userId, $filters = []) {
        $sql = "SELECT t.*, c.name as category_name, a.name as account_name
                FROM transactions t
                LEFT JOIN categories c ON t.category_id = c.id
                LEFT JOIN accounts a ON t.account_id = a.id
                WHERE t.user_id = ? AND t.deleted_at IS NULL";
        
        $params = [$userId];
        $types = 'i';

        // Filtro por tipo
        if (!empty($filters['type'])) {
            $sql .= " AND t.type = ?";
            $params[] = $filters['type'];
            $types .= 's';
        }

        // Filtro por conta
        if (!empty($filters['account_id'])) {
            $sql .= " AND t.account_id = ?";
            $params[] = $filters['account_id'];
            $types .= 'i';
        }

        // Filtro por categoria
        if (!empty($filters['category_id'])) {
            $sql .= " AND t.category_id = ?";
            $params[] = $filters['category_id'];
            $types .= 'i';
        }

        // Filtro por status
        if (!empty($filters['status'])) {
            $sql .= " AND t.status = ?";
            $params[] = $filters['status'];
            $types .= 's';
        }

        // Filtro por data inicial
        if (!empty($filters['start_date'])) {
            $sql .= " AND t.transaction_date >= ?";
            $params[] = $filters['start_date'];
            $types .= 's';
        }

        // Filtro por data final
        if (!empty($filters['end_date'])) {
            $sql .= " AND t.transaction_date <= ?";
            $params[] = $filters['end_date'];
            $types .= 's';
        }

        $sql .= " ORDER BY t.transaction_date DESC, t.created_at DESC";

        // Paginação
        $limit = $filters['limit'] ?? 20;
        $offset = ($filters['page'] ?? 1 - 1) * $limit;
        $sql .= " LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        $types .= 'ii';

        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter resumo de transações por período
     */
    public function getSummary($userId, $startDate, $endDate) {
        $sql = "SELECT 
                    SUM(CASE WHEN type = 'income' THEN amount ELSE 0 END) as total_income,
                    SUM(CASE WHEN type = 'expense' THEN amount ELSE 0 END) as total_expense,
                    COUNT(*) as total_transactions
                FROM transactions
                WHERE user_id = ? 
                    AND transaction_date >= ? 
                    AND transaction_date <= ? 
                    AND deleted_at IS NULL";
        
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('iss', $userId, $startDate, $endDate);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }

    /**
     * Obter transações recentes
     */
    public function getRecent($userId, $limit = 10) {
        $sql = "SELECT t.*, c.name as category_name, a.name as account_name
                FROM transactions t
                LEFT JOIN categories c ON t.category_id = c.id
                LEFT JOIN accounts a ON t.account_id = a.id
                WHERE t.user_id = ? AND t.deleted_at IS NULL
                ORDER BY t.transaction_date DESC, t.created_at DESC
                LIMIT ?";
        
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $userId, $limit);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter transações por categoria
     */
    public function getByCategory($userId, $categoryId, $startDate = null, $endDate = null) {
        $sql = "SELECT * FROM transactions 
                WHERE user_id = ? AND category_id = ? AND deleted_at IS NULL";
        
        $params = [$userId, $categoryId];
        $types = 'ii';

        if ($startDate) {
            $sql .= " AND transaction_date >= ?";
            $params[] = $startDate;
            $types .= 's';
        }

        if ($endDate) {
            $sql .= " AND transaction_date <= ?";
            $params[] = $endDate;
            $types .= 's';
        }

        $sql .= " ORDER BY transaction_date DESC";

        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Atualizar transação
     */
    public function update($id, $userId, $data) {
        $allowedFields = [
            'category_id', 'description', 'amount', 'transaction_date',
            'due_date', 'status', 'payment_method', 'notes'
        ];

        $updates = [];
        $params = [];
        $types = '';

        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updates[] = "$field = ?";
                $params[] = $data[$field];
                
                if (in_array($field, ['category_id'])) {
                    $types .= 'i';
                } elseif (in_array($field, ['amount'])) {
                    $types .= 'd';
                } else {
                    $types .= 's';
                }
            }
        }

        if (empty($updates)) {
            return false;
        }

        $params[] = $id;
        $params[] = $userId;
        $types .= 'ii';

        $sql = "UPDATE transactions SET " . implode(', ', $updates) . " WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);

        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        
        if (!$stmt->execute()) {
            throw new Exception("Erro ao atualizar transação: " . $stmt->error);
        }

        return true;
    }

    /**
     * Deletar transação (soft delete)
     */
    public function delete($id, $userId) {
        $sql = "UPDATE transactions SET deleted_at = NOW() WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $id, $userId);
        
        return $stmt->execute();
    }
}
?>
