<?php
/**
 * Modelo de Tag
 */

class Tag {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Criar nova tag
     */
    public function create($userId, $data) {
        $sql = "INSERT INTO tags (
            user_id, name, parent_tag_id, description, color, icon, is_active
        ) VALUES (?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param(
            'isisissi',
            $userId,
            $data['name'],
            $data['parent_tag_id'] ?? null,
            $data['description'] ?? null,
            $data['color'] ?? '#FFA500',
            $data['icon'] ?? 'tag',
            $data['is_active'] ?? 1
        );

        if (!$stmt->execute()) {
            throw new Exception("Erro ao criar tag: " . $stmt->error);
        }

        return $this->db->insert_id;
    }

    /**
     * Obter tag por ID
     */
    public function getById($id, $userId) {
        $sql = "SELECT * FROM tags WHERE id = ? AND user_id = ? AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $id, $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }

    /**
     * Obter todas as tags do usuário
     */
    public function getByUserId($userId, $onlyActive = false, $parentTagId = null) {
        $sql = "SELECT * FROM tags WHERE user_id = ? AND deleted_at IS NULL";
        
        $params = [$userId];
        $types = 'i';

        if ($onlyActive) {
            $sql .= " AND is_active = 1";
        }

        if ($parentTagId !== null) {
            $sql .= " AND parent_tag_id = ?";
            $params[] = $parentTagId;
            $types .= 'i';
        } else {
            $sql .= " AND parent_tag_id IS NULL";
        }
        
        $sql .= " ORDER BY name ASC";
        
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter subtags de uma tag
     */
    public function getSubtags($tagId, $userId, $onlyActive = false) {
        $sql = "SELECT * FROM tags WHERE parent_tag_id = ? AND user_id = ? AND deleted_at IS NULL";
        
        if ($onlyActive) {
            $sql .= " AND is_active = 1";
        }
        
        $sql .= " ORDER BY name ASC";
        
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $tagId, $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter hierarquia completa de tags
     */
    public function getHierarchy($userId, $onlyActive = false) {
        $tags = $this->getByUserId($userId, $onlyActive, null);
        
        foreach ($tags as &$tag) {
            $tag['subtags'] = $this->getSubtags($tag['id'], $userId, $onlyActive);
        }
        
        return $tags;
    }

    /**
     * Atualizar tag
     */
    public function update($id, $userId, $data) {
        $allowedFields = [
            'name', 'parent_tag_id', 'description', 'color', 'icon', 'is_active'
        ];

        $updates = [];
        $params = [];
        $types = '';

        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updates[] = "$field = ?";
                $params[] = $data[$field];
                
                if (in_array($field, ['parent_tag_id', 'is_active'])) {
                    $types .= 'i';
                } else {
                    $types .= 's';
                }
            }
        }

        if (empty($updates)) {
            return false;
        }

        $params[] = $id;
        $params[] = $userId;
        $types .= 'ii';

        $sql = "UPDATE tags SET " . implode(', ', $updates) . " WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);

        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        
        if (!$stmt->execute()) {
            throw new Exception("Erro ao atualizar tag: " . $stmt->error);
        }

        return true;
    }

    /**
     * Deletar tag (soft delete)
     */
    public function delete($id, $userId) {
        $sql = "UPDATE tags SET deleted_at = NOW() WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $id, $userId);
        
        return $stmt->execute();
    }

    /**
     * Adicionar tag a uma transação
     */
    public function addToTransaction($transactionId, $tagId) {
        $sql = "INSERT INTO transaction_tags (transaction_id, tag_id) VALUES (?, ?)";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $transactionId, $tagId);
        
        if (!$stmt->execute()) {
            throw new Exception("Erro ao adicionar tag à transação: " . $stmt->error);
        }

        return true;
    }

    /**
     * Remover tag de uma transação
     */
    public function removeFromTransaction($transactionId, $tagId) {
        $sql = "DELETE FROM transaction_tags WHERE transaction_id = ? AND tag_id = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $transactionId, $tagId);
        
        return $stmt->execute();
    }

    /**
     * Obter tags de uma transação
     */
    public function getTransactionTags($transactionId) {
        $sql = "SELECT t.* FROM tags t
                INNER JOIN transaction_tags tt ON t.id = tt.tag_id
                WHERE tt.transaction_id = ? AND t.deleted_at IS NULL
                ORDER BY t.name ASC";
        
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $transactionId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }
}
?>
