<?php
/**
 * Modelo de Relatório
 */

class Report {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obter transações com filtros avançados
     */
    public function getTransactionsWithFilters($userId, $filters = []) {
        $sql = "SELECT t.*, 
                c.name as category_name, 
                a.name as account_name,
                GROUP_CONCAT(tag.name SEPARATOR ', ') as tags
                FROM transactions t
                LEFT JOIN categories c ON t.category_id = c.id
                LEFT JOIN accounts a ON t.account_id = a.id
                LEFT JOIN transaction_tags tt ON t.id = tt.transaction_id
                LEFT JOIN tags tag ON tt.tag_id = tag.id
                WHERE t.user_id = ? AND t.deleted_at IS NULL";

        $params = [$userId];
        $types = 'i';

        // Filtro por tipo
        if (!empty($filters['type'])) {
            $sql .= " AND t.type = ?";
            $params[] = $filters['type'];
            $types .= 's';
        }

        // Filtro por conta
        if (!empty($filters['account_id'])) {
            $sql .= " AND t.account_id = ?";
            $params[] = $filters['account_id'];
            $types .= 'i';
        }

        // Filtro por categoria
        if (!empty($filters['category_id'])) {
            $sql .= " AND t.category_id = ?";
            $params[] = $filters['category_id'];
            $types .= 'i';
        }

        // Filtro por tag
        if (!empty($filters['tag_id'])) {
            $sql .= " AND t.id IN (SELECT transaction_id FROM transaction_tags WHERE tag_id = ?)";
            $params[] = $filters['tag_id'];
            $types .= 'i';
        }

        // Filtro por data inicial
        if (!empty($filters['start_date'])) {
            $sql .= " AND DATE(t.transaction_date) >= ?";
            $params[] = $filters['start_date'];
            $types .= 's';
        }

        // Filtro por data final
        if (!empty($filters['end_date'])) {
            $sql .= " AND DATE(t.transaction_date) <= ?";
            $params[] = $filters['end_date'];
            $types .= 's';
        }

        // Filtro por status
        if (!empty($filters['status'])) {
            $sql .= " AND t.status = ?";
            $params[] = $filters['status'];
            $types .= 's';
        }

        // Filtro por descrição
        if (!empty($filters['description'])) {
            $sql .= " AND t.description LIKE ?";
            $params[] = '%' . $filters['description'] . '%';
            $types .= 's';
        }

        $sql .= " GROUP BY t.id";

        // Ordenação
        $orderBy = $filters['order_by'] ?? 'transaction_date';
        $orderDir = $filters['order_dir'] ?? 'DESC';
        $sql .= " ORDER BY t.$orderBy $orderDir";

        // Paginação
        $limit = $filters['limit'] ?? 100;
        $offset = ($filters['page'] ?? 1 - 1) * $limit;
        $sql .= " LIMIT $limit OFFSET $offset";

        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter resumo de transações por período
     */
    public function getSummaryByPeriod($userId, $startDate, $endDate, $groupBy = 'day') {
        $dateFormat = match($groupBy) {
            'month' => '%Y-%m',
            'year' => '%Y',
            'week' => '%Y-W%w',
            default => '%Y-%m-%d'
        };

        $sql = "SELECT 
                DATE_FORMAT(t.transaction_date, '$dateFormat') as period,
                t.type,
                COUNT(*) as count,
                SUM(t.amount) as total
                FROM transactions t
                WHERE t.user_id = ? 
                AND t.deleted_at IS NULL
                AND DATE(t.transaction_date) >= ?
                AND DATE(t.transaction_date) <= ?
                GROUP BY period, t.type
                ORDER BY period ASC";

        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('iss', $userId, $startDate, $endDate);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter resumo por categoria
     */
    public function getSummaryByCategory($userId, $startDate, $endDate, $type = null) {
        $sql = "SELECT 
                c.id,
                c.name,
                c.color,
                COUNT(t.id) as count,
                SUM(t.amount) as total
                FROM categories c
                LEFT JOIN transactions t ON c.id = t.category_id 
                AND t.user_id = ? 
                AND t.deleted_at IS NULL
                AND DATE(t.transaction_date) >= ?
                AND DATE(t.transaction_date) <= ?";

        $params = [$userId, $startDate, $endDate];
        $types = 'iss';

        if ($type) {
            $sql .= " WHERE c.type = ?";
            $params[] = $type;
            $types .= 's';
        }

        $sql .= " GROUP BY c.id, c.name, c.color
                 HAVING total > 0
                 ORDER BY total DESC";

        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter resumo por conta
     */
    public function getSummaryByAccount($userId, $startDate, $endDate) {
        $sql = "SELECT 
                a.id,
                a.name,
                a.color,
                COUNT(t.id) as count,
                SUM(CASE WHEN t.type = 'income' THEN t.amount ELSE 0 END) as income,
                SUM(CASE WHEN t.type = 'expense' THEN t.amount ELSE 0 END) as expense
                FROM accounts a
                LEFT JOIN transactions t ON a.id = t.account_id 
                AND t.user_id = ? 
                AND t.deleted_at IS NULL
                AND DATE(t.transaction_date) >= ?
                AND DATE(t.transaction_date) <= ?
                WHERE a.user_id = ?
                GROUP BY a.id, a.name, a.color
                ORDER BY a.name ASC";

        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('issi', $userId, $startDate, $endDate, $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter resumo por tag
     */
    public function getSummaryByTag($userId, $startDate, $endDate) {
        $sql = "SELECT 
                tag.id,
                tag.name,
                tag.color,
                COUNT(DISTINCT t.id) as count,
                SUM(t.amount) as total
                FROM tags tag
                LEFT JOIN transaction_tags tt ON tag.id = tt.tag_id
                LEFT JOIN transactions t ON tt.transaction_id = t.id
                AND t.user_id = ? 
                AND t.deleted_at IS NULL
                AND DATE(t.transaction_date) >= ?
                AND DATE(t.transaction_date) <= ?
                WHERE tag.user_id = ?
                GROUP BY tag.id, tag.name, tag.color
                HAVING total > 0
                ORDER BY total DESC";

        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('issi', $userId, $startDate, $endDate, $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter totais gerais
     */
    public function getTotals($userId, $startDate, $endDate) {
        $sql = "SELECT 
                SUM(CASE WHEN t.type = 'income' THEN t.amount ELSE 0 END) as total_income,
                SUM(CASE WHEN t.type = 'expense' THEN t.amount ELSE 0 END) as total_expense,
                COUNT(*) as total_transactions
                FROM transactions t
                WHERE t.user_id = ? 
                AND t.deleted_at IS NULL
                AND DATE(t.transaction_date) >= ?
                AND DATE(t.transaction_date) <= ?";

        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('iss', $userId, $startDate, $endDate);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }
}
?>
