<?php
/**
 * Modelo de Integração Open Banking
 */

class OpenBanking {
    private $db;
    private $apiKey;
    private $apiBaseUrl;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->apiKey = getenv('OPEN_BANKING_API_KEY') ?: '';
        $this->apiBaseUrl = getenv('OPEN_BANKING_API_URL') ?: 'https://api.openbanking.example.com';
    }

    /**
     * Criar conexão com banco
     */
    public function createConnection($userId, $data) {
        $sql = "INSERT INTO open_banking_connections (
            user_id, bank_code, connection_token, account_number, account_type,
            institution_name, is_active
        ) VALUES (?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param(
            'isssssi',
            $userId,
            $data['bank_code'],
            $data['connection_token'],
            $data['account_number'],
            $data['account_type'],
            $data['institution_name'],
            $data['is_active'] ?? 1
        );

        if (!$stmt->execute()) {
            throw new Exception("Erro ao criar conexão: " . $stmt->error);
        }

        return $this->db->insert_id;
    }

    /**
     * Obter conexões do usuário
     */
    public function getConnections($userId) {
        $sql = "SELECT * FROM open_banking_connections 
                WHERE user_id = ? AND deleted_at IS NULL
                ORDER BY created_at DESC";
        
        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter conexão por ID
     */
    public function getConnectionById($id, $userId) {
        $sql = "SELECT * FROM open_banking_connections 
                WHERE id = ? AND user_id = ? AND deleted_at IS NULL";
        
        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $id, $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }

    /**
     * Sincronizar transações de um banco
     */
    public function syncTransactions($connectionId, $userId, $startDate = null, $endDate = null) {
        try {
            // Obter conexão
            $connection = $this->getConnectionById($connectionId, $userId);
            if (!$connection) {
                throw new Exception("Conexão não encontrada");
            }

            // Buscar transações da API
            $transactions = $this->fetchTransactionsFromAPI(
                $connection['connection_token'],
                $startDate,
                $endDate
            );

            // Processar e salvar transações
            $imported = 0;
            foreach ($transactions as $transaction) {
                if ($this->importTransaction($userId, $connection, $transaction)) {
                    $imported++;
                }
            }

            // Atualizar data da última sincronização
            $this->updateLastSync($connectionId);

            return [
                'success' => true,
                'imported' => $imported,
                'total' => count($transactions)
            ];
        } catch (Exception $e) {
            throw new Exception("Erro ao sincronizar: " . $e->getMessage());
        }
    }

    /**
     * Buscar transações da API
     */
    private function fetchTransactionsFromAPI($token, $startDate = null, $endDate = null) {
        // Esta é uma implementação exemplo
        // Adapte para a API específica do seu provedor de Open Banking
        
        $url = $this->apiBaseUrl . '/transactions';
        
        $params = [
            'token' => $token,
            'start_date' => $startDate,
            'end_date' => $endDate
        ];

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url . '?' . http_build_query($params),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->apiKey,
                'Content-Type: application/json'
            ],
            CURLOPT_TIMEOUT => 30
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200) {
            throw new Exception("Erro ao buscar transações: HTTP $httpCode");
        }

        $data = json_decode($response, true);
        return $data['transactions'] ?? [];
    }

    /**
     * Importar transação
     */
    private function importTransaction($userId, $connection, $transactionData) {
        try {
            // Verificar se a transação já existe
            $existingSql = "SELECT id FROM transactions 
                           WHERE user_id = ? AND external_id = ? AND deleted_at IS NULL";
            $stmt = $this->db->prepare($existingSql);
            $stmt->bind_param('is', $userId, $transactionData['id']);
            $stmt->execute();
            
            if ($stmt->get_result()->fetch_assoc()) {
                return false; // Transação já existe
            }

            // Obter ou criar conta
            $accountId = $this->getOrCreateAccount($userId, $connection, $transactionData);

            // Obter ou criar categoria
            $categoryId = $this->getOrCreateCategory($userId, $transactionData);

            // Inserir transação
            $sql = "INSERT INTO transactions (
                user_id, account_id, category_id, type, amount, description,
                transaction_date, status, external_id, external_source
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

            $stmt = $this->db->prepare($sql);
            if (!$stmt) {
                throw new Exception("Erro ao preparar query");
            }

            $type = $transactionData['amount'] > 0 ? 'income' : 'expense';
            $amount = abs($transactionData['amount']);
            $status = 'completed';
            $externalSource = 'open_banking';

            $stmt->bind_param(
                'iiisdsssss',
                $userId,
                $accountId,
                $categoryId,
                $type,
                $amount,
                $transactionData['description'],
                $transactionData['date'],
                $status,
                $transactionData['id'],
                $externalSource
            );

            return $stmt->execute();
        } catch (Exception $e) {
            logError('Erro ao importar transação: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Obter ou criar conta
     */
    private function getOrCreateAccount($userId, $connection, $transactionData) {
        // Verificar se a conta já existe
        $sql = "SELECT id FROM accounts 
                WHERE user_id = ? AND external_id = ? AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        $externalId = $connection['id'] . '_' . $transactionData['account_id'];
        $stmt->bind_param('is', $userId, $externalId);
        $stmt->execute();
        
        $result = $stmt->get_result()->fetch_assoc();
        if ($result) {
            return $result['id'];
        }

        // Criar nova conta
        $accountModel = new Account();
        $accountData = [
            'name' => $connection['institution_name'] . ' - ' . $transactionData['account_name'],
            'account_type' => 'checking',
            'bank_name' => $connection['institution_name'],
            'balance' => 0,
            'external_id' => $externalId,
            'external_source' => 'open_banking'
        ];

        return $accountModel->create($userId, $accountData);
    }

    /**
     * Obter ou criar categoria
     */
    private function getOrCreateCategory($userId, $transactionData) {
        $categoryName = $transactionData['category'] ?? 'Importado';
        $type = $transactionData['amount'] > 0 ? 'income' : 'expense';

        // Verificar se a categoria já existe
        $sql = "SELECT id FROM categories 
                WHERE user_id = ? AND name = ? AND type = ? AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        $stmt->bind_param('iss', $userId, $categoryName, $type);
        $stmt->execute();
        
        $result = $stmt->get_result()->fetch_assoc();
        if ($result) {
            return $result['id'];
        }

        // Criar nova categoria
        $categoryModel = new Category();
        $categoryData = [
            'name' => $categoryName,
            'type' => $type,
            'color' => '#FFA500'
        ];

        return $categoryModel->create($userId, $categoryData);
    }

    /**
     * Atualizar data da última sincronização
     */
    private function updateLastSync($connectionId) {
        $sql = "UPDATE open_banking_connections 
                SET last_sync_at = NOW() 
                WHERE id = ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->bind_param('i', $connectionId);
        $stmt->execute();
    }

    /**
     * Deletar conexão
     */
    public function deleteConnection($id, $userId) {
        $sql = "UPDATE open_banking_connections 
                SET deleted_at = NOW() 
                WHERE id = ? AND user_id = ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->bind_param('ii', $id, $userId);
        
        return $stmt->execute();
    }

    /**
     * Obter status de sincronização
     */
    public function getSyncStatus($connectionId, $userId) {
        $sql = "SELECT id, bank_code, institution_name, is_active, 
                last_sync_at, created_at
                FROM open_banking_connections 
                WHERE id = ? AND user_id = ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->bind_param('ii', $connectionId, $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }
}
?>
