<?php
/**
 * Modelo de Cartão de Crédito
 */

class CreditCard {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Criar novo cartão
     */
    public function create($userId, $data) {
        $sql = "INSERT INTO credit_cards (
            user_id, account_id, name, card_number, card_holder, expiry_month,
            expiry_year, cvv, card_type, bank_name, credit_limit, current_balance,
            due_day, color, icon, is_active
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param(
            'iisssiiissddiissi',
            $userId,
            $data['account_id'] ?? null,
            $data['name'],
            $data['card_number'] ?? null,
            $data['card_holder'] ?? null,
            $data['expiry_month'] ?? null,
            $data['expiry_year'] ?? null,
            $data['cvv'] ?? null,
            $data['card_type'] ?? 'credit',
            $data['bank_name'] ?? null,
            $data['credit_limit'] ?? 0,
            $data['current_balance'] ?? 0,
            $data['due_day'] ?? 10,
            $data['color'] ?? '#1A3A52',
            $data['icon'] ?? 'credit-card',
            $data['is_active'] ?? 1
        );

        if (!$stmt->execute()) {
            throw new Exception("Erro ao criar cartão: " . $stmt->error);
        }

        return $this->db->insert_id;
    }

    /**
     * Obter cartão por ID
     */
    public function getById($id, $userId) {
        $sql = "SELECT * FROM credit_cards WHERE id = ? AND user_id = ? AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $id, $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }

    /**
     * Obter todos os cartões do usuário
     */
    public function getByUserId($userId, $onlyActive = false) {
        $sql = "SELECT * FROM credit_cards WHERE user_id = ? AND deleted_at IS NULL";
        
        if ($onlyActive) {
            $sql .= " AND is_active = 1";
        }
        
        $sql .= " ORDER BY name ASC";
        
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter limite total disponível
     */
    public function getTotalCreditLimit($userId) {
        $sql = "SELECT SUM(credit_limit) as total FROM credit_cards WHERE user_id = ? AND is_active = 1 AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $userId);
        $stmt->execute();
        
        $result = $stmt->get_result()->fetch_assoc();
        return $result['total'] ?? 0;
    }

    /**
     * Obter saldo total dos cartões
     */
    public function getTotalBalance($userId) {
        $sql = "SELECT SUM(current_balance) as total FROM credit_cards WHERE user_id = ? AND is_active = 1 AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $userId);
        $stmt->execute();
        
        $result = $stmt->get_result()->fetch_assoc();
        return $result['total'] ?? 0;
    }

    /**
     * Atualizar saldo do cartão
     */
    public function updateBalance($id, $amount, $userId) {
        $sql = "UPDATE credit_cards SET current_balance = current_balance + ? WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('dii', $amount, $id, $userId);
        
        if (!$stmt->execute()) {
            throw new Exception("Erro ao atualizar saldo: " . $stmt->error);
        }

        return true;
    }

    /**
     * Atualizar cartão
     */
    public function update($id, $userId, $data) {
        $allowedFields = [
            'account_id', 'name', 'card_number', 'card_holder', 'expiry_month',
            'expiry_year', 'cvv', 'card_type', 'bank_name', 'credit_limit',
            'current_balance', 'due_day', 'color', 'icon', 'is_active'
        ];

        $updates = [];
        $params = [];
        $types = '';

        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updates[] = "$field = ?";
                $params[] = $data[$field];
                
                if (in_array($field, ['account_id', 'expiry_month', 'expiry_year', 'due_day', 'is_active'])) {
                    $types .= 'i';
                } elseif (in_array($field, ['credit_limit', 'current_balance'])) {
                    $types .= 'd';
                } else {
                    $types .= 's';
                }
            }
        }

        if (empty($updates)) {
            return false;
        }

        $params[] = $id;
        $params[] = $userId;
        $types .= 'ii';

        $sql = "UPDATE credit_cards SET " . implode(', ', $updates) . " WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);

        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        
        if (!$stmt->execute()) {
            throw new Exception("Erro ao atualizar cartão: " . $stmt->error);
        }

        return true;
    }

    /**
     * Deletar cartão (soft delete)
     */
    public function delete($id, $userId) {
        $sql = "UPDATE credit_cards SET deleted_at = NOW() WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $id, $userId);
        
        return $stmt->execute();
    }
}
?>
