<?php
/**
 * Modelo de Categoria
 */

class Category {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Criar nova categoria
     */
    public function create($userId, $data) {
        $sql = "INSERT INTO categories (
            user_id, name, description, type, color, icon, is_active
        ) VALUES (?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param(
            'isssssi',
            $userId,
            $data['name'],
            $data['description'] ?? null,
            $data['type'],
            $data['color'] ?? '#2D5016',
            $data['icon'] ?? 'folder',
            $data['is_active'] ?? 1
        );

        if (!$stmt->execute()) {
            throw new Exception("Erro ao criar categoria: " . $stmt->error);
        }

        return $this->db->insert_id;
    }

    /**
     * Obter categoria por ID
     */
    public function getById($id, $userId) {
        $sql = "SELECT * FROM categories WHERE id = ? AND user_id = ? AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $id, $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }

    /**
     * Obter categorias por tipo
     */
    public function getByType($userId, $type, $onlyActive = false) {
        $sql = "SELECT * FROM categories WHERE user_id = ? AND type = ? AND deleted_at IS NULL";
        
        if ($onlyActive) {
            $sql .= " AND is_active = 1";
        }
        
        $sql .= " ORDER BY name ASC";
        
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('is', $userId, $type);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter todas as categorias do usuário
     */
    public function getByUserId($userId, $onlyActive = false) {
        $sql = "SELECT * FROM categories WHERE user_id = ? AND deleted_at IS NULL";
        
        if ($onlyActive) {
            $sql .= " AND is_active = 1";
        }
        
        $sql .= " ORDER BY type ASC, name ASC";
        
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Atualizar categoria
     */
    public function update($id, $userId, $data) {
        $allowedFields = [
            'name', 'description', 'color', 'icon', 'is_active'
        ];

        $updates = [];
        $params = [];
        $types = '';

        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updates[] = "$field = ?";
                $params[] = $data[$field];
                
                if (in_array($field, ['is_active'])) {
                    $types .= 'i';
                } else {
                    $types .= 's';
                }
            }
        }

        if (empty($updates)) {
            return false;
        }

        $params[] = $id;
        $params[] = $userId;
        $types .= 'ii';

        $sql = "UPDATE categories SET " . implode(', ', $updates) . " WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);

        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        
        if (!$stmt->execute()) {
            throw new Exception("Erro ao atualizar categoria: " . $stmt->error);
        }

        return true;
    }

    /**
     * Deletar categoria (soft delete)
     */
    public function delete($id, $userId) {
        $sql = "UPDATE categories SET deleted_at = NOW() WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $id, $userId);
        
        return $stmt->execute();
    }

    /**
     * Criar categorias padrão para novo usuário
     */
    public function createDefaultCategories($userId) {
        $defaultCategories = [
            // Receitas
            ['name' => 'Salário', 'type' => 'income', 'icon' => 'briefcase', 'color' => '#2D5016'],
            ['name' => 'Freelance', 'type' => 'income', 'icon' => 'laptop', 'color' => '#2D5016'],
            ['name' => 'Investimentos', 'type' => 'income', 'icon' => 'trending-up', 'color' => '#2D5016'],
            ['name' => 'Outros', 'type' => 'income', 'icon' => 'plus-circle', 'color' => '#2D5016'],
            
            // Despesas
            ['name' => 'Alimentação', 'type' => 'expense', 'icon' => 'shopping-cart', 'color' => '#8B0000'],
            ['name' => 'Transporte', 'type' => 'expense', 'icon' => 'truck', 'color' => '#8B0000'],
            ['name' => 'Moradia', 'type' => 'expense', 'icon' => 'home', 'color' => '#8B0000'],
            ['name' => 'Saúde', 'type' => 'expense', 'icon' => 'heart', 'color' => '#8B0000'],
            ['name' => 'Educação', 'type' => 'expense', 'icon' => 'book', 'color' => '#8B0000'],
            ['name' => 'Lazer', 'type' => 'expense', 'icon' => 'smile', 'color' => '#8B0000'],
            ['name' => 'Utilidades', 'type' => 'expense', 'icon' => 'zap', 'color' => '#8B0000'],
            ['name' => 'Outros', 'type' => 'expense', 'icon' => 'minus-circle', 'color' => '#8B0000'],
        ];

        foreach ($defaultCategories as $category) {
            try {
                $this->create($userId, $category);
            } catch (Exception $e) {
                logError('Erro ao criar categoria padrão: ' . $e->getMessage());
            }
        }

        return true;
    }
}
?>
