<?php
/**
 * Modelo de Conta Bancária
 */

class Account {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Criar nova conta
     */
    public function create($userId, $data) {
        $sql = "INSERT INTO accounts (
            user_id, name, account_type, bank_name, account_number,
            agency_number, balance, currency, color, icon, is_active, is_default
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->db->prepare($sql);
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param(
            'isssssdsssii',
            $userId,
            $data['name'],
            $data['account_type'] ?? 'checking',
            $data['bank_name'] ?? null,
            $data['account_number'] ?? null,
            $data['agency_number'] ?? null,
            $data['balance'] ?? 0,
            $data['currency'] ?? 'BRL',
            $data['color'] ?? '#2D5016',
            $data['icon'] ?? 'bank',
            $data['is_active'] ?? 1,
            $data['is_default'] ?? 0
        );

        if (!$stmt->execute()) {
            throw new Exception("Erro ao criar conta: " . $stmt->error);
        }

        return $this->db->insert_id;
    }

    /**
     * Obter conta por ID
     */
    public function getById($id, $userId) {
        $sql = "SELECT * FROM accounts WHERE id = ? AND user_id = ? AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $id, $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_assoc();
    }

    /**
     * Obter todas as contas do usuário
     */
    public function getByUserId($userId, $onlyActive = false) {
        $sql = "SELECT * FROM accounts WHERE user_id = ? AND deleted_at IS NULL";
        
        if ($onlyActive) {
            $sql .= " AND is_active = 1";
        }
        
        $sql .= " ORDER BY is_default DESC, name ASC";
        
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $userId);
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    /**
     * Obter saldo total do usuário
     */
    public function getTotalBalance($userId) {
        $sql = "SELECT SUM(balance) as total FROM accounts WHERE user_id = ? AND is_active = 1 AND deleted_at IS NULL";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('i', $userId);
        $stmt->execute();
        
        $result = $stmt->get_result()->fetch_assoc();
        return $result['total'] ?? 0;
    }

    /**
     * Atualizar saldo da conta
     */
    public function updateBalance($id, $amount, $userId) {
        $sql = "UPDATE accounts SET balance = balance + ? WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('dii', $amount, $id, $userId);
        
        if (!$stmt->execute()) {
            throw new Exception("Erro ao atualizar saldo: " . $stmt->error);
        }

        return true;
    }

    /**
     * Atualizar conta
     */
    public function update($id, $userId, $data) {
        $allowedFields = [
            'name', 'account_type', 'bank_name', 'account_number',
            'agency_number', 'balance', 'color', 'icon', 'is_active', 'is_default'
        ];

        $updates = [];
        $params = [];
        $types = '';

        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updates[] = "$field = ?";
                $params[] = $data[$field];
                
                if (in_array($field, ['is_active', 'is_default'])) {
                    $types .= 'i';
                } elseif (in_array($field, ['balance'])) {
                    $types .= 'd';
                } else {
                    $types .= 's';
                }
            }
        }

        if (empty($updates)) {
            return false;
        }

        $params[] = $id;
        $params[] = $userId;
        $types .= 'ii';

        $sql = "UPDATE accounts SET " . implode(', ', $updates) . " WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);

        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param($types, ...$params);
        
        if (!$stmt->execute()) {
            throw new Exception("Erro ao atualizar conta: " . $stmt->error);
        }

        return true;
    }

    /**
     * Deletar conta (soft delete)
     */
    public function delete($id, $userId) {
        $sql = "UPDATE accounts SET deleted_at = NOW() WHERE id = ? AND user_id = ?";
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Erro ao preparar query: " . $this->db->error);
        }

        $stmt->bind_param('ii', $id, $userId);
        
        return $stmt->execute();
    }
}
?>
