<?php
/**
 * Classe Auxiliar APP
 * Funções globais da aplicação
 */

class AppHelper {
    /**
     * Formatar moeda
     */
    public static function formatCurrency($value, $currency = 'BRL') {
        $formatter = new NumberFormatter('pt_BR', NumberFormatter::CURRENCY);
        return $formatter->formatCurrency($value, $currency);
    }

    /**
     * Formatar data
     */
    public static function formatDate($date, $format = 'd/m/Y') {
        if (empty($date)) {
            return '';
        }

        $timestamp = strtotime($date);
        return date($format, $timestamp);
    }

    /**
     * Formatar data e hora
     */
    public static function formatDateTime($date, $format = 'd/m/Y H:i') {
        if (empty($date)) {
            return '';
        }

        $timestamp = strtotime($date);
        return date($format, $timestamp);
    }

    /**
     * Calcular diferença de datas em dias
     */
    public static function daysDifference($date1, $date2) {
        $timestamp1 = strtotime($date1);
        $timestamp2 = strtotime($date2);
        $diff = abs($timestamp2 - $timestamp1);
        return floor($diff / (60 * 60 * 24));
    }

    /**
     * Gerar slug a partir de texto
     */
    public static function generateSlug($text) {
        $text = strtolower($text);
        $text = preg_replace('/[^a-z0-9]+/', '-', $text);
        $text = trim($text, '-');
        return $text;
    }

    /**
     * Gerar token aleatório
     */
    public static function generateToken($length = 32) {
        return bin2hex(random_bytes($length / 2));
    }

    /**
     * Validar email
     */
    public static function isValidEmail($email) {
        return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
    }

    /**
     * Sanitizar entrada
     */
    public static function sanitize($input) {
        return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
    }

    /**
     * Validar força de senha
     */
    public static function validatePasswordStrength($password) {
        $strength = 0;
        $feedback = [];

        if (strlen($password) >= 8) {
            $strength++;
        } else {
            $feedback[] = 'Senha deve ter no mínimo 8 caracteres';
        }

        if (preg_match('/[a-z]/', $password)) {
            $strength++;
        } else {
            $feedback[] = 'Adicione letras minúsculas';
        }

        if (preg_match('/[A-Z]/', $password)) {
            $strength++;
        } else {
            $feedback[] = 'Adicione letras maiúsculas';
        }

        if (preg_match('/[0-9]/', $password)) {
            $strength++;
        } else {
            $feedback[] = 'Adicione números';
        }

        if (preg_match('/[!@#$%^&*()_+\-=\[\]{};\':"\\|,.<>\/?]/', $password)) {
            $strength++;
        } else {
            $feedback[] = 'Adicione caracteres especiais';
        }

        return [
            'strength' => $strength,
            'feedback' => $feedback,
            'score' => ($strength / 5) * 100
        ];
    }

    /**
     * Calcular percentual
     */
    public static function calculatePercentage($value, $total) {
        if ($total == 0) {
            return 0;
        }
        return ($value / $total) * 100;
    }

    /**
     * Formatar número
     */
    public static function formatNumber($number, $decimals = 2) {
        return number_format($number, $decimals, ',', '.');
    }

    /**
     * Obter iniciais do nome
     */
    public static function getInitials($name) {
        $parts = explode(' ', $name);
        $initials = '';
        
        foreach ($parts as $part) {
            if (!empty($part)) {
                $initials .= strtoupper($part[0]);
            }
        }
        
        return substr($initials, 0, 2);
    }

    /**
     * Gerar cor aleatória
     */
    public static function generateRandomColor() {
        return '#' . str_pad(dechex(mt_rand(0, 0xFFFFFF)), 6, '0', STR_PAD_LEFT);
    }

    /**
     * Converter para JSON seguro
     */
    public static function toJSON($data) {
        return json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    /**
     * Paginar array
     */
    public static function paginate($array, $page = 1, $perPage = 20) {
        $total = count($array);
        $pages = ceil($total / $perPage);
        $page = max(1, min($page, $pages));
        $start = ($page - 1) * $perPage;

        return [
            'data' => array_slice($array, $start, $perPage),
            'current_page' => $page,
            'total_pages' => $pages,
            'total_items' => $total,
            'per_page' => $perPage
        ];
    }

    /**
     * Obter período de data formatado
     */
    public static function getDatePeriodLabel($startDate, $endDate) {
        $start = new DateTime($startDate);
        $end = new DateTime($endDate);

        if ($start->format('Y-m') === $end->format('Y-m')) {
            return $start->format('M/Y');
        }

        return $start->format('d/m/Y') . ' - ' . $end->format('d/m/Y');
    }

    /**
     * Calcular idade
     */
    public static function calculateAge($birthDate) {
        $birth = new DateTime($birthDate);
        $today = new DateTime();
        $age = $today->diff($birth);
        return $age->y;
    }

    /**
     * Formatar tempo decorrido
     */
    public static function timeAgo($date) {
        $timestamp = strtotime($date);
        $time_difference = time() - $timestamp;

        if ($time_difference < 60) {
            return 'Agora';
        } elseif ($time_difference < 3600) {
            $minutes = floor($time_difference / 60);
            return $minutes . ' minuto' . ($minutes > 1 ? 's' : '') . ' atrás';
        } elseif ($time_difference < 86400) {
            $hours = floor($time_difference / 3600);
            return $hours . ' hora' . ($hours > 1 ? 's' : '') . ' atrás';
        } elseif ($time_difference < 604800) {
            $days = floor($time_difference / 86400);
            return $days . ' dia' . ($days > 1 ? 's' : '') . ' atrás';
        } else {
            return date('d/m/Y', $timestamp);
        }
    }
}

// Criar alias global APP
class APP extends AppHelper {}
?>
