<?php
/**
 * Controlador de Transações
 */

require_once __DIR__ . '/../models/Transaction.php';
require_once __DIR__ . '/../models/Account.php';

class TransactionController {
    private $transactionModel;
    private $accountModel;

    public function __construct() {
        $this->transactionModel = new Transaction();
        $this->accountModel = new Account();
    }

    /**
     * Criar transação
     */
    public function create() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $data = json_decode(file_get_contents('php://input'), true);

        // Validações
        if (empty($data['account_id']) || !is_numeric($data['account_id'])) {
            http_response_code(400);
            return json_encode(['error' => 'Conta inválida']);
        }

        if (empty($data['type']) || !in_array($data['type'], ['income', 'expense', 'transfer'])) {
            http_response_code(400);
            return json_encode(['error' => 'Tipo de transação inválido']);
        }

        if (empty($data['amount']) || !is_numeric($data['amount']) || $data['amount'] <= 0) {
            http_response_code(400);
            return json_encode(['error' => 'Valor inválido']);
        }

        // Verificar se a conta pertence ao usuário
        $account = $this->accountModel->getById($data['account_id'], $userId);
        if (!$account) {
            http_response_code(403);
            return json_encode(['error' => 'Conta não encontrada']);
        }

        try {
            // Criar transação
            $transactionId = $this->transactionModel->create($userId, $data);

            // Atualizar saldo da conta
            $amountChange = $data['type'] === 'income' ? $data['amount'] : -$data['amount'];
            $this->accountModel->updateBalance($data['account_id'], $amountChange, $userId);

            http_response_code(201);
            return json_encode([
                'success' => true,
                'message' => 'Transação criada com sucesso',
                'transaction_id' => $transactionId
            ]);
        } catch (Exception $e) {
            logError('Erro ao criar transação: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao criar transação']);
        }
    }

    /**
     * Obter transações
     */
    public function getList() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        
        $filters = [
            'type' => $_GET['type'] ?? null,
            'account_id' => $_GET['account_id'] ?? null,
            'category_id' => $_GET['category_id'] ?? null,
            'status' => $_GET['status'] ?? null,
            'start_date' => $_GET['start_date'] ?? null,
            'end_date' => $_GET['end_date'] ?? null,
            'page' => $_GET['page'] ?? 1,
            'limit' => $_GET['limit'] ?? 20
        ];

        try {
            $transactions = $this->transactionModel->getByUserId($userId, $filters);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'transactions' => $transactions
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter transações: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter transações']);
        }
    }

    /**
     * Obter transação por ID
     */
    public function getById() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $transactionId = $_GET['id'] ?? null;

        if (!$transactionId || !is_numeric($transactionId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de transação inválido']);
        }

        try {
            $transaction = $this->transactionModel->getById($transactionId, $userId);

            if (!$transaction) {
                http_response_code(404);
                return json_encode(['error' => 'Transação não encontrada']);
            }

            http_response_code(200);
            return json_encode([
                'success' => true,
                'transaction' => $transaction
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter transação: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter transação']);
        }
    }

    /**
     * Atualizar transação
     */
    public function update() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'PUT') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $transactionId = $_GET['id'] ?? null;
        $data = json_decode(file_get_contents('php://input'), true);

        if (!$transactionId || !is_numeric($transactionId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de transação inválido']);
        }

        try {
            // Verificar se a transação pertence ao usuário
            $transaction = $this->transactionModel->getById($transactionId, $userId);
            if (!$transaction) {
                http_response_code(404);
                return json_encode(['error' => 'Transação não encontrada']);
            }

            // Atualizar transação
            $this->transactionModel->update($transactionId, $userId, $data);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Transação atualizada com sucesso'
            ]);
        } catch (Exception $e) {
            logError('Erro ao atualizar transação: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao atualizar transação']);
        }
    }

    /**
     * Deletar transação
     */
    public function delete() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'DELETE') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $transactionId = $_GET['id'] ?? null;

        if (!$transactionId || !is_numeric($transactionId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de transação inválido']);
        }

        try {
            // Verificar se a transação pertence ao usuário
            $transaction = $this->transactionModel->getById($transactionId, $userId);
            if (!$transaction) {
                http_response_code(404);
                return json_encode(['error' => 'Transação não encontrada']);
            }

            // Reverter o saldo da conta
            $amountChange = $transaction['type'] === 'income' ? -$transaction['amount'] : $transaction['amount'];
            $this->accountModel->updateBalance($transaction['account_id'], $amountChange, $userId);

            // Deletar transação
            $this->transactionModel->delete($transactionId, $userId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Transação deletada com sucesso'
            ]);
        } catch (Exception $e) {
            logError('Erro ao deletar transação: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao deletar transação']);
        }
    }
}
?>
