<?php
/**
 * Controlador de Tags
 */

require_once __DIR__ . '/../models/Tag.php';

class TagController {
    private $tagModel;

    public function __construct() {
        $this->tagModel = new Tag();
    }

    /**
     * Criar tag
     */
    public function create() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $data = json_decode(file_get_contents('php://input'), true);

        // Validações
        if (empty($data['name'])) {
            http_response_code(400);
            return json_encode(['error' => 'Nome da tag é obrigatório']);
        }

        try {
            $tagId = $this->tagModel->create($userId, $data);

            http_response_code(201);
            return json_encode([
                'success' => true,
                'message' => 'Tag criada com sucesso',
                'tag_id' => $tagId
            ]);
        } catch (Exception $e) {
            logError('Erro ao criar tag: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao criar tag']);
        }
    }

    /**
     * Obter tags
     */
    public function getList() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $onlyActive = $_GET['active'] ?? false;
        $hierarchy = $_GET['hierarchy'] ?? false;

        try {
            if ($hierarchy) {
                $tags = $this->tagModel->getHierarchy($userId, $onlyActive);
            } else {
                $tags = $this->tagModel->getByUserId($userId, $onlyActive);
            }

            http_response_code(200);
            return json_encode([
                'success' => true,
                'tags' => $tags
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter tags: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter tags']);
        }
    }

    /**
     * Obter tag por ID
     */
    public function getById() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $tagId = $_GET['id'] ?? null;

        if (!$tagId || !is_numeric($tagId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de tag inválido']);
        }

        try {
            $tag = $this->tagModel->getById($tagId, $userId);

            if (!$tag) {
                http_response_code(404);
                return json_encode(['error' => 'Tag não encontrada']);
            }

            // Obter subtags
            $tag['subtags'] = $this->tagModel->getSubtags($tagId, $userId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'tag' => $tag
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter tag: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter tag']);
        }
    }

    /**
     * Atualizar tag
     */
    public function update() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'PUT') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $tagId = $_GET['id'] ?? null;
        $data = json_decode(file_get_contents('php://input'), true);

        if (!$tagId || !is_numeric($tagId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de tag inválido']);
        }

        try {
            // Verificar se a tag pertence ao usuário
            $tag = $this->tagModel->getById($tagId, $userId);
            if (!$tag) {
                http_response_code(404);
                return json_encode(['error' => 'Tag não encontrada']);
            }

            // Atualizar tag
            $this->tagModel->update($tagId, $userId, $data);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Tag atualizada com sucesso'
            ]);
        } catch (Exception $e) {
            logError('Erro ao atualizar tag: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao atualizar tag']);
        }
    }

    /**
     * Deletar tag
     */
    public function delete() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'DELETE') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $tagId = $_GET['id'] ?? null;

        if (!$tagId || !is_numeric($tagId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de tag inválido']);
        }

        try {
            // Verificar se a tag pertence ao usuário
            $tag = $this->tagModel->getById($tagId, $userId);
            if (!$tag) {
                http_response_code(404);
                return json_encode(['error' => 'Tag não encontrada']);
            }

            // Deletar tag
            $this->tagModel->delete($tagId, $userId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Tag deletada com sucesso'
            ]);
        } catch (Exception $e) {
            logError('Erro ao deletar tag: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao deletar tag']);
        }
    }

    /**
     * Adicionar tag a uma transação
     */
    public function addToTransaction() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $data = json_decode(file_get_contents('php://input'), true);

        if (empty($data['transaction_id']) || empty($data['tag_id'])) {
            http_response_code(400);
            return json_encode(['error' => 'Parâmetros inválidos']);
        }

        try {
            $this->tagModel->addToTransaction($data['transaction_id'], $data['tag_id']);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Tag adicionada à transação'
            ]);
        } catch (Exception $e) {
            logError('Erro ao adicionar tag: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao adicionar tag']);
        }
    }

    /**
     * Remover tag de uma transação
     */
    public function removeFromTransaction() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'DELETE') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $transactionId = $_GET['transaction_id'] ?? null;
        $tagId = $_GET['tag_id'] ?? null;

        if (!$transactionId || !$tagId) {
            http_response_code(400);
            return json_encode(['error' => 'Parâmetros inválidos']);
        }

        try {
            $this->tagModel->removeFromTransaction($transactionId, $tagId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Tag removida da transação'
            ]);
        } catch (Exception $e) {
            logError('Erro ao remover tag: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao remover tag']);
        }
    }

    /**
     * Obter tags de uma transação
     */
    public function getTransactionTags() {
        AuthController::requireAuth();
        
        $transactionId = $_GET['transaction_id'] ?? null;

        if (!$transactionId || !is_numeric($transactionId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de transação inválido']);
        }

        try {
            $tags = $this->tagModel->getTransactionTags($transactionId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'tags' => $tags
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter tags da transação: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter tags']);
        }
    }
}
?>
