<?php
/**
 * Controlador de Relatórios
 */

require_once __DIR__ . '/../models/Report.php';

class ReportController {
    private $reportModel;

    public function __construct() {
        $this->reportModel = new Report();
    }

    /**
     * Obter relatório com filtros
     */
    public function getReport() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        
        $filters = [
            'type' => $_GET['type'] ?? null,
            'account_id' => $_GET['account_id'] ?? null,
            'category_id' => $_GET['category_id'] ?? null,
            'tag_id' => $_GET['tag_id'] ?? null,
            'start_date' => $_GET['start_date'] ?? date('Y-m-01'),
            'end_date' => $_GET['end_date'] ?? date('Y-m-d'),
            'status' => $_GET['status'] ?? null,
            'description' => $_GET['description'] ?? null,
            'order_by' => $_GET['order_by'] ?? 'transaction_date',
            'order_dir' => $_GET['order_dir'] ?? 'DESC',
            'page' => $_GET['page'] ?? 1,
            'limit' => $_GET['limit'] ?? 100
        ];

        try {
            $transactions = $this->reportModel->getTransactionsWithFilters($userId, $filters);
            $totals = $this->reportModel->getTotals($userId, $filters['start_date'], $filters['end_date']);
            $summaryByCategory = $this->reportModel->getSummaryByCategory($userId, $filters['start_date'], $filters['end_date']);
            $summaryByAccount = $this->reportModel->getSummaryByAccount($userId, $filters['start_date'], $filters['end_date']);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'transactions' => $transactions,
                'totals' => $totals,
                'summary_by_category' => $summaryByCategory,
                'summary_by_account' => $summaryByAccount,
                'filters' => $filters
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter relatório: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter relatório']);
        }
    }

    /**
     * Obter resumo por período
     */
    public function getSummaryByPeriod() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $startDate = $_GET['start_date'] ?? date('Y-m-01');
        $endDate = $_GET['end_date'] ?? date('Y-m-d');
        $groupBy = $_GET['group_by'] ?? 'day';

        try {
            $summary = $this->reportModel->getSummaryByPeriod($userId, $startDate, $endDate, $groupBy);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'summary' => $summary,
                'group_by' => $groupBy
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter resumo: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter resumo']);
        }
    }

    /**
     * Obter resumo por categoria
     */
    public function getSummaryByCategory() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $startDate = $_GET['start_date'] ?? date('Y-m-01');
        $endDate = $_GET['end_date'] ?? date('Y-m-d');
        $type = $_GET['type'] ?? null;

        try {
            $summary = $this->reportModel->getSummaryByCategory($userId, $startDate, $endDate, $type);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'summary' => $summary
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter resumo por categoria: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter resumo']);
        }
    }

    /**
     * Obter resumo por conta
     */
    public function getSummaryByAccount() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $startDate = $_GET['start_date'] ?? date('Y-m-01');
        $endDate = $_GET['end_date'] ?? date('Y-m-d');

        try {
            $summary = $this->reportModel->getSummaryByAccount($userId, $startDate, $endDate);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'summary' => $summary
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter resumo por conta: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter resumo']);
        }
    }

    /**
     * Obter resumo por tag
     */
    public function getSummaryByTag() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $startDate = $_GET['start_date'] ?? date('Y-m-01');
        $endDate = $_GET['end_date'] ?? date('Y-m-d');

        try {
            $summary = $this->reportModel->getSummaryByTag($userId, $startDate, $endDate);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'summary' => $summary
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter resumo por tag: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter resumo']);
        }
    }

    /**
     * Exportar relatório em PDF
     */
    public function exportPDF() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        
        $filters = [
            'type' => $_GET['type'] ?? null,
            'account_id' => $_GET['account_id'] ?? null,
            'category_id' => $_GET['category_id'] ?? null,
            'tag_id' => $_GET['tag_id'] ?? null,
            'start_date' => $_GET['start_date'] ?? date('Y-m-01'),
            'end_date' => $_GET['end_date'] ?? date('Y-m-d'),
            'status' => $_GET['status'] ?? null,
            'limit' => 1000
        ];

        try {
            $transactions = $this->reportModel->getTransactionsWithFilters($userId, $filters);
            $totals = $this->reportModel->getTotals($userId, $filters['start_date'], $filters['end_date']);
            $summaryByCategory = $this->reportModel->getSummaryByCategory($userId, $filters['start_date'], $filters['end_date']);

            // Gerar PDF
            require_once __DIR__ . '/../helpers/PDFGenerator.php';
            $pdfGenerator = new PDFGenerator();
            
            $html = $this->generatePDFHTML($transactions, $totals, $summaryByCategory, $filters);
            $pdfGenerator->generate($html, 'relatorio_financeiro_' . date('Y-m-d') . '.pdf');

        } catch (Exception $e) {
            logError('Erro ao exportar PDF: ' . $e->getMessage());
            http_response_code(500);
            echo json_encode(['error' => 'Erro ao exportar PDF']);
        }
    }

    /**
     * Gerar HTML para PDF
     */
    private function generatePDFHTML($transactions, $totals, $summaryByCategory, $filters) {
        $html = '
        <!DOCTYPE html>
        <html lang="pt-BR">
        <head>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; margin: 20px; }
                h1 { color: #1A3A52; text-align: center; }
                h2 { color: #2D5016; margin-top: 20px; border-bottom: 2px solid #2D5016; padding-bottom: 10px; }
                table { width: 100%; border-collapse: collapse; margin: 20px 0; }
                th { background-color: #2D5016; color: white; padding: 10px; text-align: left; }
                td { padding: 8px; border-bottom: 1px solid #ddd; }
                tr:hover { background-color: #f5f5f5; }
                .summary { background-color: #f9f9f9; padding: 15px; margin: 20px 0; border-radius: 5px; }
                .positive { color: #2D5016; font-weight: bold; }
                .negative { color: #8B0000; font-weight: bold; }
                .footer { text-align: center; margin-top: 30px; color: #666; font-size: 12px; }
            </style>
        </head>
        <body>
            <h1>Relatório Financeiro</h1>
            <p style="text-align: center; color: #666;">
                Período: ' . date('d/m/Y', strtotime($filters['start_date'])) . ' a ' . date('d/m/Y', strtotime($filters['end_date'])) . '
            </p>

            <h2>Resumo</h2>
            <div class="summary">
                <p><strong>Total de Receitas:</strong> <span class="positive">R$ ' . number_format($totals['total_income'] ?? 0, 2, ',', '.') . '</span></p>
                <p><strong>Total de Despesas:</strong> <span class="negative">R$ ' . number_format($totals['total_expense'] ?? 0, 2, ',', '.') . '</span></p>
                <p><strong>Saldo:</strong> <span class="' . (($totals['total_income'] - $totals['total_expense']) >= 0 ? 'positive' : 'negative') . '">R$ ' . number_format($totals['total_income'] - $totals['total_expense'], 2, ',', '.') . '</span></p>
                <p><strong>Total de Transações:</strong> ' . ($totals['total_transactions'] ?? 0) . '</p>
            </div>

            <h2>Resumo por Categoria</h2>
            <table>
                <thead>
                    <tr>
                        <th>Categoria</th>
                        <th>Quantidade</th>
                        <th>Total</th>
                    </tr>
                </thead>
                <tbody>';

        foreach ($summaryByCategory as $category) {
            $html .= '
                    <tr>
                        <td>' . htmlspecialchars($category['name']) . '</td>
                        <td>' . $category['count'] . '</td>
                        <td>R$ ' . number_format($category['total'] ?? 0, 2, ',', '.') . '</td>
                    </tr>';
        }

        $html .= '
                </tbody>
            </table>

            <h2>Transações</h2>
            <table>
                <thead>
                    <tr>
                        <th>Data</th>
                        <th>Descrição</th>
                        <th>Categoria</th>
                        <th>Conta</th>
                        <th>Tipo</th>
                        <th>Valor</th>
                    </tr>
                </thead>
                <tbody>';

        foreach ($transactions as $transaction) {
            $html .= '
                    <tr>
                        <td>' . date('d/m/Y', strtotime($transaction['transaction_date'])) . '</td>
                        <td>' . htmlspecialchars($transaction['description'] ?? '-') . '</td>
                        <td>' . htmlspecialchars($transaction['category_name'] ?? '-') . '</td>
                        <td>' . htmlspecialchars($transaction['account_name'] ?? '-') . '</td>
                        <td>' . ($transaction['type'] === 'income' ? 'Receita' : 'Despesa') . '</td>
                        <td class="' . ($transaction['type'] === 'income' ? 'positive' : 'negative') . '">R$ ' . number_format($transaction['amount'], 2, ',', '.') . '</td>
                    </tr>';
        }

        $html .= '
                </tbody>
            </table>

            <div class="footer">
                <p>Relatório gerado em ' . date('d/m/Y H:i:s') . '</p>
            </div>
        </body>
        </html>';

        return $html;
    }
}
?>
