<?php
/**
 * Controlador de Open Banking
 */

require_once __DIR__ . '/../models/OpenBanking.php';

class OpenBankingController {
    private $openBankingModel;

    public function __construct() {
        $this->openBankingModel = new OpenBanking();
    }

    /**
     * Obter conexões
     */
    public function getConnections() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();

        try {
            $connections = $this->openBankingModel->getConnections($userId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'connections' => $connections
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter conexões: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter conexões']);
        }
    }

    /**
     * Criar conexão
     */
    public function createConnection() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $data = json_decode(file_get_contents('php://input'), true);

        // Validações
        if (empty($data['bank_code']) || empty($data['connection_token'])) {
            http_response_code(400);
            return json_encode(['error' => 'Dados de conexão inválidos']);
        }

        try {
            $connectionId = $this->openBankingModel->createConnection($userId, $data);

            http_response_code(201);
            return json_encode([
                'success' => true,
                'message' => 'Conexão criada com sucesso',
                'connection_id' => $connectionId
            ]);
        } catch (Exception $e) {
            logError('Erro ao criar conexão: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao criar conexão']);
        }
    }

    /**
     * Sincronizar transações
     */
    public function syncTransactions() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $connectionId = $_GET['connection_id'] ?? null;
        $data = json_decode(file_get_contents('php://input'), true);

        if (!$connectionId || !is_numeric($connectionId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de conexão inválido']);
        }

        try {
            $result = $this->openBankingModel->syncTransactions(
                $connectionId,
                $userId,
                $data['start_date'] ?? null,
                $data['end_date'] ?? null
            );

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Sincronização concluída',
                'imported' => $result['imported'],
                'total' => $result['total']
            ]);
        } catch (Exception $e) {
            logError('Erro ao sincronizar: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao sincronizar transações']);
        }
    }

    /**
     * Obter status de sincronização
     */
    public function getSyncStatus() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $connectionId = $_GET['connection_id'] ?? null;

        if (!$connectionId || !is_numeric($connectionId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de conexão inválido']);
        }

        try {
            $status = $this->openBankingModel->getSyncStatus($connectionId, $userId);

            if (!$status) {
                http_response_code(404);
                return json_encode(['error' => 'Conexão não encontrada']);
            }

            http_response_code(200);
            return json_encode([
                'success' => true,
                'status' => $status
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter status: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter status']);
        }
    }

    /**
     * Deletar conexão
     */
    public function deleteConnection() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'DELETE') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $connectionId = $_GET['connection_id'] ?? null;

        if (!$connectionId || !is_numeric($connectionId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de conexão inválido']);
        }

        try {
            $this->openBankingModel->deleteConnection($connectionId, $userId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Conexão deletada com sucesso'
            ]);
        } catch (Exception $e) {
            logError('Erro ao deletar conexão: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao deletar conexão']);
        }
    }

    /**
     * Obter URL de autorização para conectar banco
     */
    public function getAuthorizationURL() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $bankCode = $_GET['bank_code'] ?? null;

        if (!$bankCode) {
            http_response_code(400);
            return json_encode(['error' => 'Código do banco é obrigatório']);
        }

        try {
            // Gerar URL de autorização (implementar conforme sua API)
            $redirectUri = BASE_URL . '/open-banking/callback';
            $state = bin2hex(random_bytes(16));
            
            // Salvar state na sessão para validação
            $_SESSION['open_banking_state'] = $state;
            $_SESSION['open_banking_bank_code'] = $bankCode;

            // Construir URL de autorização
            $authUrl = 'https://api.openbanking.example.com/authorize?' . http_build_query([
                'client_id' => getenv('OPEN_BANKING_CLIENT_ID'),
                'redirect_uri' => $redirectUri,
                'state' => $state,
                'bank_code' => $bankCode,
                'response_type' => 'code'
            ]);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'authorization_url' => $authUrl
            ]);
        } catch (Exception $e) {
            logError('Erro ao gerar URL de autorização: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao gerar URL de autorização']);
        }
    }
}
?>
