<?php
/**
 * Controlador do Dashboard
 */

require_once __DIR__ . '/../models/User.php';
require_once __DIR__ . '/../models/Account.php';
require_once __DIR__ . '/../models/Transaction.php';

class DashboardController {
    private $userModel;
    private $accountModel;
    private $transactionModel;

    public function __construct() {
        $this->userModel = new User();
        $this->accountModel = new Account();
        $this->transactionModel = new Transaction();
    }

    /**
     * Obter dados do dashboard
     */
    public function getDashboardData() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();

        try {
            $user = $this->userModel->getById($userId);
            
            if (!$user) {
                http_response_code(404);
                return json_encode(['error' => 'Usuário não encontrado']);
            }

            // Obter contas
            $accounts = $this->accountModel->getByUserId($userId, true);
            $totalBalance = $this->accountModel->getTotalBalance($userId);

            // Período atual (mês)
            $startDate = date('Y-m-01');
            $endDate = date('Y-m-d');

            // Obter resumo de transações
            $summary = $this->transactionModel->getSummary($userId, $startDate, $endDate);

            // Obter transações recentes
            $recentTransactions = $this->transactionModel->getRecent($userId, 10);

            // Calcular saldo disponível
            $availableBalance = $totalBalance;

            // Preparar resposta
            $data = [
                'success' => true,
                'user' => [
                    'id' => $user['id'],
                    'name' => $user['first_name'] . ' ' . $user['last_name'],
                    'email' => $user['email'],
                    'currency' => $user['currency']
                ],
                'summary' => [
                    'total_balance' => floatval($totalBalance),
                    'total_income' => floatval($summary['total_income'] ?? 0),
                    'total_expense' => floatval($summary['total_expense'] ?? 0),
                    'available_balance' => floatval($availableBalance),
                    'total_transactions' => intval($summary['total_transactions'] ?? 0),
                    'period' => [
                        'start_date' => $startDate,
                        'end_date' => $endDate
                    ]
                ],
                'accounts' => $accounts,
                'recent_transactions' => $recentTransactions
            ];

            http_response_code(200);
            return json_encode($data);
        } catch (Exception $e) {
            logError('Erro ao obter dados do dashboard: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter dados do dashboard']);
        }
    }

    /**
     * Obter dados para gráficos
     */
    public function getChartData() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();

        try {
            // Período (últimos 30 dias)
            $endDate = date('Y-m-d');
            $startDate = date('Y-m-d', strtotime('-30 days'));

            // Obter transações por dia
            $sql = "SELECT 
                        DATE(transaction_date) as date,
                        SUM(CASE WHEN type = 'income' THEN amount ELSE 0 END) as income,
                        SUM(CASE WHEN type = 'expense' THEN amount ELSE 0 END) as expense
                    FROM transactions
                    WHERE user_id = ? 
                        AND transaction_date >= ? 
                        AND transaction_date <= ? 
                        AND deleted_at IS NULL
                    GROUP BY DATE(transaction_date)
                    ORDER BY transaction_date ASC";

            $db = Database::getInstance();
            $stmt = $db->prepare($sql);
            $stmt->bind_param('iss', $userId, $startDate, $endDate);
            $stmt->execute();
            
            $chartData = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

            // Obter distribuição por categoria
            $sql = "SELECT 
                        c.name as category,
                        SUM(t.amount) as total,
                        COUNT(*) as count
                    FROM transactions t
                    LEFT JOIN categories c ON t.category_id = c.id
                    WHERE t.user_id = ? 
                        AND t.type = 'expense'
                        AND t.transaction_date >= ? 
                        AND t.transaction_date <= ? 
                        AND t.deleted_at IS NULL
                    GROUP BY t.category_id
                    ORDER BY total DESC
                    LIMIT 10";

            $stmt = $db->prepare($sql);
            $stmt->bind_param('iss', $userId, $startDate, $endDate);
            $stmt->execute();
            
            $categoryData = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

            $data = [
                'success' => true,
                'chart_data' => $chartData,
                'category_data' => $categoryData,
                'period' => [
                    'start_date' => $startDate,
                    'end_date' => $endDate
                ]
            ];

            http_response_code(200);
            return json_encode($data);
        } catch (Exception $e) {
            logError('Erro ao obter dados dos gráficos: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter dados dos gráficos']);
        }
    }
}
?>
