<?php
/**
 * Controlador de Cartões de Crédito
 */

require_once __DIR__ . '/../models/CreditCard.php';

class CreditCardController {
    private $creditCardModel;

    public function __construct() {
        $this->creditCardModel = new CreditCard();
    }

    /**
     * Criar cartão
     */
    public function create() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $data = json_decode(file_get_contents('php://input'), true);

        // Validações
        if (empty($data['name'])) {
            http_response_code(400);
            return json_encode(['error' => 'Nome do cartão é obrigatório']);
        }

        if (empty($data['card_type']) || !in_array($data['card_type'], ['credit', 'debit', 'prepaid'])) {
            http_response_code(400);
            return json_encode(['error' => 'Tipo de cartão inválido']);
        }

        try {
            $cardId = $this->creditCardModel->create($userId, $data);

            http_response_code(201);
            return json_encode([
                'success' => true,
                'message' => 'Cartão criado com sucesso',
                'card_id' => $cardId
            ]);
        } catch (Exception $e) {
            logError('Erro ao criar cartão: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao criar cartão']);
        }
    }

    /**
     * Obter cartões
     */
    public function getList() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $onlyActive = $_GET['active'] ?? false;

        try {
            $cards = $this->creditCardModel->getByUserId($userId, $onlyActive);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'cards' => $cards
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter cartões: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter cartões']);
        }
    }

    /**
     * Obter cartão por ID
     */
    public function getById() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $cardId = $_GET['id'] ?? null;

        if (!$cardId || !is_numeric($cardId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de cartão inválido']);
        }

        try {
            $card = $this->creditCardModel->getById($cardId, $userId);

            if (!$card) {
                http_response_code(404);
                return json_encode(['error' => 'Cartão não encontrado']);
            }

            http_response_code(200);
            return json_encode([
                'success' => true,
                'card' => $card
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter cartão: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter cartão']);
        }
    }

    /**
     * Atualizar cartão
     */
    public function update() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'PUT') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $cardId = $_GET['id'] ?? null;
        $data = json_decode(file_get_contents('php://input'), true);

        if (!$cardId || !is_numeric($cardId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de cartão inválido']);
        }

        try {
            // Verificar se o cartão pertence ao usuário
            $card = $this->creditCardModel->getById($cardId, $userId);
            if (!$card) {
                http_response_code(404);
                return json_encode(['error' => 'Cartão não encontrado']);
            }

            // Atualizar cartão
            $this->creditCardModel->update($cardId, $userId, $data);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Cartão atualizado com sucesso'
            ]);
        } catch (Exception $e) {
            logError('Erro ao atualizar cartão: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao atualizar cartão']);
        }
    }

    /**
     * Deletar cartão
     */
    public function delete() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'DELETE') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $cardId = $_GET['id'] ?? null;

        if (!$cardId || !is_numeric($cardId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de cartão inválido']);
        }

        try {
            // Verificar se o cartão pertence ao usuário
            $card = $this->creditCardModel->getById($cardId, $userId);
            if (!$card) {
                http_response_code(404);
                return json_encode(['error' => 'Cartão não encontrado']);
            }

            // Deletar cartão
            $this->creditCardModel->delete($cardId, $userId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Cartão deletado com sucesso'
            ]);
        } catch (Exception $e) {
            logError('Erro ao deletar cartão: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao deletar cartão']);
        }
    }

    /**
     * Obter limite total disponível
     */
    public function getTotalCreditLimit() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();

        try {
            $totalLimit = $this->creditCardModel->getTotalCreditLimit($userId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'total_credit_limit' => floatval($totalLimit)
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter limite total: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter limite total']);
        }
    }

    /**
     * Obter saldo total dos cartões
     */
    public function getTotalBalance() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();

        try {
            $totalBalance = $this->creditCardModel->getTotalBalance($userId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'total_balance' => floatval($totalBalance)
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter saldo total: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter saldo total']);
        }
    }
}
?>
