<?php
/**
 * Controlador de Categorias
 */

require_once __DIR__ . '/../models/Category.php';

class CategoryController {
    private $categoryModel;

    public function __construct() {
        $this->categoryModel = new Category();
    }

    /**
     * Criar categoria
     */
    public function create() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $data = json_decode(file_get_contents('php://input'), true);

        // Validações
        if (empty($data['name'])) {
            http_response_code(400);
            return json_encode(['error' => 'Nome da categoria é obrigatório']);
        }

        if (empty($data['type']) || !in_array($data['type'], ['income', 'expense'])) {
            http_response_code(400);
            return json_encode(['error' => 'Tipo de categoria inválido']);
        }

        try {
            $categoryId = $this->categoryModel->create($userId, $data);

            http_response_code(201);
            return json_encode([
                'success' => true,
                'message' => 'Categoria criada com sucesso',
                'category_id' => $categoryId
            ]);
        } catch (Exception $e) {
            logError('Erro ao criar categoria: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao criar categoria']);
        }
    }

    /**
     * Obter categorias
     */
    public function getList() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $type = $_GET['type'] ?? null;
        $onlyActive = $_GET['active'] ?? false;

        try {
            if ($type) {
                $categories = $this->categoryModel->getByType($userId, $type, $onlyActive);
            } else {
                $categories = $this->categoryModel->getByUserId($userId, $onlyActive);
            }

            http_response_code(200);
            return json_encode([
                'success' => true,
                'categories' => $categories
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter categorias: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter categorias']);
        }
    }

    /**
     * Obter categoria por ID
     */
    public function getById() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $categoryId = $_GET['id'] ?? null;

        if (!$categoryId || !is_numeric($categoryId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de categoria inválido']);
        }

        try {
            $category = $this->categoryModel->getById($categoryId, $userId);

            if (!$category) {
                http_response_code(404);
                return json_encode(['error' => 'Categoria não encontrada']);
            }

            http_response_code(200);
            return json_encode([
                'success' => true,
                'category' => $category
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter categoria: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter categoria']);
        }
    }

    /**
     * Atualizar categoria
     */
    public function update() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'PUT') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $categoryId = $_GET['id'] ?? null;
        $data = json_decode(file_get_contents('php://input'), true);

        if (!$categoryId || !is_numeric($categoryId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de categoria inválido']);
        }

        try {
            // Verificar se a categoria pertence ao usuário
            $category = $this->categoryModel->getById($categoryId, $userId);
            if (!$category) {
                http_response_code(404);
                return json_encode(['error' => 'Categoria não encontrada']);
            }

            // Atualizar categoria
            $this->categoryModel->update($categoryId, $userId, $data);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Categoria atualizada com sucesso'
            ]);
        } catch (Exception $e) {
            logError('Erro ao atualizar categoria: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao atualizar categoria']);
        }
    }

    /**
     * Deletar categoria
     */
    public function delete() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'DELETE') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $categoryId = $_GET['id'] ?? null;

        if (!$categoryId || !is_numeric($categoryId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de categoria inválido']);
        }

        try {
            // Verificar se a categoria pertence ao usuário
            $category = $this->categoryModel->getById($categoryId, $userId);
            if (!$category) {
                http_response_code(404);
                return json_encode(['error' => 'Categoria não encontrada']);
            }

            // Deletar categoria
            $this->categoryModel->delete($categoryId, $userId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Categoria deletada com sucesso'
            ]);
        } catch (Exception $e) {
            logError('Erro ao deletar categoria: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao deletar categoria']);
        }
    }
}
?>
