<?php
/**
 * Controlador de Autenticação
 */

require_once __DIR__ . '/../models/User.php';

class AuthController {
    private $userModel;

    public function __construct() {
        $this->userModel = new User();
    }

    /**
     * Registrar novo usuário
     */
    public function register() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $data = json_decode(file_get_contents('php://input'), true);

        // Validações
        if (empty($data['email']) || !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            http_response_code(400);
            return json_encode(['error' => 'Email inválido']);
        }

        if (empty($data['password']) || strlen($data['password']) < 8) {
            http_response_code(400);
            return json_encode(['error' => 'Senha deve ter no mínimo 8 caracteres']);
        }

        if ($data['password'] !== $data['password_confirm']) {
            http_response_code(400);
            return json_encode(['error' => 'Senhas não conferem']);
        }

        // Verificar se email já existe
        $existingUser = $this->userModel->getByEmail($data['email']);
        if ($existingUser) {
            http_response_code(409);
            return json_encode(['error' => 'Email já cadastrado']);
        }

        try {
            $userId = $this->userModel->create([
                'email' => $data['email'],
                'password' => $data['password'],
                'first_name' => $data['first_name'] ?? '',
                'last_name' => $data['last_name'] ?? '',
                'auth_provider' => 'email'
            ]);

            // Criar sessão
            $_SESSION['user_id'] = $userId;
            $_SESSION['email'] = $data['email'];

            http_response_code(201);
            return json_encode([
                'success' => true,
                'message' => 'Usuário registrado com sucesso',
                'user_id' => $userId
            ]);
        } catch (Exception $e) {
            logError('Erro ao registrar usuário: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao registrar usuário']);
        }
    }

    /**
     * Login com email e senha
     */
    public function login() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $data = json_decode(file_get_contents('php://input'), true);

        if (empty($data['email']) || empty($data['password'])) {
            http_response_code(400);
            return json_encode(['error' => 'Email e senha são obrigatórios']);
        }

        // Verificar se usuário está bloqueado
        if ($this->userModel->isLocked($data['email'])) {
            http_response_code(429);
            return json_encode(['error' => 'Usuário bloqueado por excesso de tentativas. Tente novamente mais tarde.']);
        }

        $user = $this->userModel->getByEmail($data['email']);

        if (!$user || !$this->userModel->verifyPassword($data['password'], $user['password_hash'])) {
            $this->userModel->incrementLoginAttempts($data['email']);
            
            if ($user) {
                $loginAttempts = $user['login_attempts'] + 1;
                if ($loginAttempts >= MAX_LOGIN_ATTEMPTS) {
                    $this->userModel->lockUser($data['email'], LOCKOUT_TIME);
                }
            }

            http_response_code(401);
            return json_encode(['error' => 'Email ou senha incorretos']);
        }

        // Reset login attempts
        $this->userModel->unlockUser($data['email']);

        // Atualizar última data de login
        $this->userModel->updateLastLogin($user['id']);

        // Criar sessão
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['email'] = $user['email'];
        $_SESSION['first_name'] = $user['first_name'];

        http_response_code(200);
        return json_encode([
            'success' => true,
            'message' => 'Login realizado com sucesso',
            'user' => [
                'id' => $user['id'],
                'email' => $user['email'],
                'first_name' => $user['first_name'],
                'last_name' => $user['last_name']
            ]
        ]);
    }

    /**
     * Login com Google
     */
    public function googleCallback() {
        if (empty($_GET['code'])) {
            http_response_code(400);
            return json_encode(['error' => 'Código de autorização não fornecido']);
        }

        try {
            // Aqui você implementaria a troca do código por um token
            // Usando a biblioteca Google PHP Client
            
            // Este é um exemplo simplificado
            $code = $_GET['code'];
            
            // Você precisará usar a biblioteca oficial do Google
            // https://github.com/googleapis/google-api-php-client
            
            http_response_code(200);
            return json_encode(['success' => true, 'message' => 'Login com Google em desenvolvimento']);
        } catch (Exception $e) {
            logError('Erro ao fazer login com Google: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao fazer login com Google']);
        }
    }

    /**
     * Login com Apple
     */
    public function appleCallback() {
        if (empty($_POST['user']) || empty($_POST['id_token'])) {
            http_response_code(400);
            return json_encode(['error' => 'Dados de autenticação não fornecidos']);
        }

        try {
            // Aqui você implementaria a verificação do token JWT do Apple
            
            http_response_code(200);
            return json_encode(['success' => true, 'message' => 'Login com Apple em desenvolvimento']);
        } catch (Exception $e) {
            logError('Erro ao fazer login com Apple: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao fazer login com Apple']);
        }
    }

    /**
     * Logout
     */
    public function logout() {
        session_destroy();
        
        http_response_code(200);
        return json_encode(['success' => true, 'message' => 'Logout realizado com sucesso']);
    }

    /**
     * Obter usuário atual
     */
    public function getCurrentUser() {
        if (empty($_SESSION['user_id'])) {
            http_response_code(401);
            return json_encode(['error' => 'Não autenticado']);
        }

        try {
            $user = $this->userModel->getById($_SESSION['user_id']);
            
            if (!$user) {
                http_response_code(404);
                return json_encode(['error' => 'Usuário não encontrado']);
            }

            unset($user['password_hash']);
            unset($user['two_factor_secret']);

            http_response_code(200);
            return json_encode(['success' => true, 'user' => $user]);
        } catch (Exception $e) {
            logError('Erro ao obter usuário atual: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter usuário']);
        }
    }

    /**
     * Atualizar perfil do usuário
     */
    public function updateProfile() {
        if ($_SERVER['REQUEST_METHOD'] !== 'PUT') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        if (empty($_SESSION['user_id'])) {
            http_response_code(401);
            return json_encode(['error' => 'Não autenticado']);
        }

        $data = json_decode(file_get_contents('php://input'), true);

        try {
            $this->userModel->update($_SESSION['user_id'], $data);

            http_response_code(200);
            return json_encode(['success' => true, 'message' => 'Perfil atualizado com sucesso']);
        } catch (Exception $e) {
            logError('Erro ao atualizar perfil: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao atualizar perfil']);
        }
    }

    /**
     * Verificar se usuário está autenticado
     */
    public static function isAuthenticated() {
        return !empty($_SESSION['user_id']);
    }

    /**
     * Obter ID do usuário autenticado
     */
    public static function getUserId() {
        return $_SESSION['user_id'] ?? null;
    }

    /**
     * Verificar autenticação (middleware)
     */
    public static function requireAuth() {
        if (!self::isAuthenticated()) {
            http_response_code(401);
            echo json_encode(['error' => 'Não autenticado']);
            exit;
        }
    }
}
?>
