<?php
/**
 * Controlador de Contas
 */

require_once __DIR__ . '/../models/Account.php';

class AccountController {
    private $accountModel;

    public function __construct() {
        $this->accountModel = new Account();
    }

    /**
     * Criar conta
     */
    public function create() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $data = json_decode(file_get_contents('php://input'), true);

        // Validações
        if (empty($data['name'])) {
            http_response_code(400);
            return json_encode(['error' => 'Nome da conta é obrigatório']);
        }

        if (empty($data['account_type']) || !in_array($data['account_type'], ['checking', 'savings', 'investment', 'credit', 'other'])) {
            http_response_code(400);
            return json_encode(['error' => 'Tipo de conta inválido']);
        }

        try {
            $accountId = $this->accountModel->create($userId, $data);

            http_response_code(201);
            return json_encode([
                'success' => true,
                'message' => 'Conta criada com sucesso',
                'account_id' => $accountId
            ]);
        } catch (Exception $e) {
            logError('Erro ao criar conta: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao criar conta']);
        }
    }

    /**
     * Obter contas
     */
    public function getList() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $onlyActive = $_GET['active'] ?? false;

        try {
            $accounts = $this->accountModel->getByUserId($userId, $onlyActive);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'accounts' => $accounts
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter contas: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter contas']);
        }
    }

    /**
     * Obter conta por ID
     */
    public function getById() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();
        $accountId = $_GET['id'] ?? null;

        if (!$accountId || !is_numeric($accountId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de conta inválido']);
        }

        try {
            $account = $this->accountModel->getById($accountId, $userId);

            if (!$account) {
                http_response_code(404);
                return json_encode(['error' => 'Conta não encontrada']);
            }

            http_response_code(200);
            return json_encode([
                'success' => true,
                'account' => $account
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter conta: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter conta']);
        }
    }

    /**
     * Atualizar conta
     */
    public function update() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'PUT') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $accountId = $_GET['id'] ?? null;
        $data = json_decode(file_get_contents('php://input'), true);

        if (!$accountId || !is_numeric($accountId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de conta inválido']);
        }

        try {
            // Verificar se a conta pertence ao usuário
            $account = $this->accountModel->getById($accountId, $userId);
            if (!$account) {
                http_response_code(404);
                return json_encode(['error' => 'Conta não encontrada']);
            }

            // Atualizar conta
            $this->accountModel->update($accountId, $userId, $data);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Conta atualizada com sucesso'
            ]);
        } catch (Exception $e) {
            logError('Erro ao atualizar conta: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao atualizar conta']);
        }
    }

    /**
     * Deletar conta
     */
    public function delete() {
        AuthController::requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'DELETE') {
            http_response_code(405);
            return json_encode(['error' => 'Método não permitido']);
        }

        $userId = AuthController::getUserId();
        $accountId = $_GET['id'] ?? null;

        if (!$accountId || !is_numeric($accountId)) {
            http_response_code(400);
            return json_encode(['error' => 'ID de conta inválido']);
        }

        try {
            // Verificar se a conta pertence ao usuário
            $account = $this->accountModel->getById($accountId, $userId);
            if (!$account) {
                http_response_code(404);
                return json_encode(['error' => 'Conta não encontrada']);
            }

            // Deletar conta
            $this->accountModel->delete($accountId, $userId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'message' => 'Conta deletada com sucesso'
            ]);
        } catch (Exception $e) {
            logError('Erro ao deletar conta: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao deletar conta']);
        }
    }

    /**
     * Obter saldo total
     */
    public function getTotalBalance() {
        AuthController::requireAuth();
        
        $userId = AuthController::getUserId();

        try {
            $totalBalance = $this->accountModel->getTotalBalance($userId);

            http_response_code(200);
            return json_encode([
                'success' => true,
                'total_balance' => floatval($totalBalance)
            ]);
        } catch (Exception $e) {
            logError('Erro ao obter saldo total: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => 'Erro ao obter saldo total']);
        }
    }
}
?>
