/**
 * Finance Manager - Transações JavaScript
 */

const Transactions = {
    accounts: [],
    transactions: [],

    // Inicializar
    async init() {
        try {
            // Carregar contas
            await this.loadAccounts();
            
            // Carregar transações
            await this.loadTransactions();
            
            // Setup de eventos
            this.setupEventListeners();
        } catch (error) {
            console.error('Erro ao inicializar transações:', error);
            APP.showNotification('Erro ao carregar transações', 'error');
        }
    },

    // Carregar contas
    async loadAccounts() {
        try {
            const response = await APP.request(
                `${APP.baseUrl}api/accounts/list`,
                { method: 'GET' }
            );

            if (response.success) {
                this.accounts = response.accounts;
                this.populateAccountSelect();
            }
        } catch (error) {
            console.error('Erro ao carregar contas:', error);
        }
    },

    // Preencher select de contas
    populateAccountSelect() {
        const select = document.getElementById('trans-account');
        if (!select) return;

        select.innerHTML = '<option value="">Selecione uma conta</option>';
        this.accounts.forEach(account => {
            const option = document.createElement('option');
            option.value = account.id;
            option.textContent = account.name;
            select.appendChild(option);
        });
    },

    // Carregar transações
    async loadTransactions() {
        try {
            const filters = this.getFilters();
            const queryString = new URLSearchParams(filters).toString();
            
            const response = await APP.request(
                `${APP.baseUrl}api/transactions/list?${queryString}`,
                { method: 'GET' }
            );

            if (response.success) {
                this.transactions = response.transactions;
                this.renderTransactions();
            }
        } catch (error) {
            console.error('Erro ao carregar transações:', error);
            APP.showNotification('Erro ao carregar transações', 'error');
        }
    },

    // Obter filtros
    getFilters() {
        return {
            type: document.getElementById('filter-type')?.value || '',
            start_date: document.getElementById('filter-start-date')?.value || '',
            end_date: document.getElementById('filter-end-date')?.value || '',
            limit: 50
        };
    },

    // Renderizar transações
    renderTransactions() {
        const tbody = document.getElementById('transactions-table');
        if (!tbody) return;

        if (this.transactions.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="8" style="text-align: center; padding: 40px;">
                        <p style="color: var(--color-medium-gray);">Nenhuma transação encontrada</p>
                    </td>
                </tr>
            `;
            return;
        }

        tbody.innerHTML = this.transactions.map(transaction => `
            <tr>
                <td>${APP.formatDate(transaction.transaction_date)}</td>
                <td>${this.escapeHtml(transaction.description || 'Transação')}</td>
                <td>${this.escapeHtml(transaction.category_name || '-')}</td>
                <td>${this.escapeHtml(transaction.account_name || '-')}</td>
                <td>
                    <span class="badge ${this.getBadgeClass(transaction.type)}">
                        ${this.getTypeLabel(transaction.type)}
                    </span>
                </td>
                <td style="text-align: right; font-weight: 600; color: ${this.getAmountColor(transaction.type)};">
                    ${(transaction.type === 'income' ? '+' : '-')} ${APP.formatCurrency(transaction.amount)}
                </td>
                <td>
                    <span class="badge ${this.getStatusBadgeClass(transaction.status)}">
                        ${this.getStatusLabel(transaction.status)}
                    </span>
                </td>
                <td>
                    <button class="btn btn-sm btn-outline" onclick="Transactions.editTransaction(${transaction.id})">
                        Editar
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="Transactions.deleteTransaction(${transaction.id})">
                        Deletar
                    </button>
                </td>
            </tr>
        `).join('');
    },

    // Setup de eventos
    setupEventListeners() {
        // Botão de filtro
        const filterBtn = document.querySelector('button[onclick="applyFilters()"]');
        if (filterBtn) {
            filterBtn.addEventListener('click', () => this.loadTransactions());
        }
    },

    // Obter classe de badge para tipo
    getBadgeClass(type) {
        switch (type) {
            case 'income':
                return 'badge-success';
            case 'expense':
                return 'badge-danger';
            case 'transfer':
                return 'badge-info';
            default:
                return 'badge-primary';
        }
    },

    // Obter rótulo de tipo
    getTypeLabel(type) {
        switch (type) {
            case 'income':
                return 'Receita';
            case 'expense':
                return 'Despesa';
            case 'transfer':
                return 'Transferência';
            default:
                return type;
        }
    },

    // Obter cor do valor
    getAmountColor(type) {
        return type === 'income' ? 'var(--color-military-green)' : 'var(--color-dark-red)';
    },

    // Obter classe de badge para status
    getStatusBadgeClass(status) {
        switch (status) {
            case 'completed':
                return 'badge-success';
            case 'pending':
                return 'badge-warning';
            case 'cancelled':
                return 'badge-danger';
            case 'scheduled':
                return 'badge-info';
            default:
                return 'badge-primary';
        }
    },

    // Obter rótulo de status
    getStatusLabel(status) {
        switch (status) {
            case 'completed':
                return 'Concluída';
            case 'pending':
                return 'Pendente';
            case 'cancelled':
                return 'Cancelada';
            case 'scheduled':
                return 'Agendada';
            default:
                return status;
        }
    },

    // Editar transação
    editTransaction(id) {
        APP.showNotification('Funcionalidade em desenvolvimento', 'info');
    },

    // Deletar transação
    async deleteTransaction(id) {
        if (!confirm('Tem certeza que deseja deletar esta transação?')) {
            return;
        }

        try {
            const response = await APP.request(
                `${APP.baseUrl}api/transactions/delete?id=${id}`,
                { method: 'DELETE' }
            );

            if (response.success) {
                APP.showNotification('Transação deletada com sucesso!', 'success');
                this.loadTransactions();
            }
        } catch (error) {
            console.error('Erro ao deletar transação:', error);
            APP.showNotification(error.message || 'Erro ao deletar transação', 'error');
        }
    },

    // Escapar HTML
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
};

// Função para aplicar filtros
function applyFilters() {
    Transactions.loadTransactions();
}

// Função para submeter formulário de transação
async function submitTransactionForm() {
    const form = document.getElementById('transaction-form');
    
    if (!APP.validateForm(form)) {
        return;
    }

    const formData = new FormData(form);
    const data = {
        type: formData.get('type'),
        description: formData.get('description'),
        amount: parseFloat(formData.get('amount')),
        account_id: parseInt(formData.get('account_id')),
        transaction_date: formData.get('transaction_date')
    };

    try {
        const response = await APP.request(
            `${APP.baseUrl}api/transactions/create`,
            {
                method: 'POST',
                body: JSON.stringify(data)
            }
        );

        if (response.success) {
            APP.showNotification('Transação registrada com sucesso!', 'success');
            APP.closeModal('add-transaction-modal');
            form.reset();
            
            // Recarregar transações
            Transactions.loadTransactions();
        }
    } catch (error) {
        console.error('Erro ao registrar transação:', error);
        APP.showNotification(error.message || 'Erro ao registrar transação', 'error');
    }
}

// Inicializar quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    Transactions.init();
});

// Exportar para uso global
window.Transactions = Transactions;
