/**
 * Finance Manager - Tags JavaScript
 */

const Tags = {
    tags: [],
    hierarchy: [],

    // Inicializar
    async init() {
        try {
            // Carregar tags
            await this.loadTags();
            
            // Setup de eventos
            this.setupEventListeners();
        } catch (error) {
            console.error('Erro ao inicializar tags:', error);
            APP.showNotification('Erro ao carregar tags', 'error');
        }
    },

    // Carregar tags
    async loadTags() {
        try {
            const response = await APP.request(
                `${APP.baseUrl}api/tags/list?hierarchy=true`,
                { method: 'GET' }
            );

            if (response.success) {
                this.hierarchy = response.tags;
                this.renderTags();
                this.populateParentTagSelect();
            }
        } catch (error) {
            console.error('Erro ao carregar tags:', error);
            APP.showNotification('Erro ao carregar tags', 'error');
        }
    },

    // Renderizar tags
    renderTags() {
        const list = document.getElementById('tags-list');
        if (!list) return;

        if (this.hierarchy.length === 0) {
            list.innerHTML = `
                <p style="color: var(--color-medium-gray); text-align: center;">
                    Nenhuma tag criada ainda
                </p>
            `;
            return;
        }

        list.innerHTML = this.hierarchy.map(tag => `
            <div style="margin-bottom: 16px; padding: 12px; border: 1px solid #e0e0e0; border-radius: 8px; border-left: 4px solid ${tag.color};">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                    <div>
                        <h4 style="margin: 0; color: var(--color-dark-navy);">
                            <span style="display: inline-block; width: 12px; height: 12px; background-color: ${tag.color}; border-radius: 50%; margin-right: 8px;"></span>
                            ${this.escapeHtml(tag.name)}
                        </h4>
                        ${tag.description ? `<p style="margin: 4px 0 0 0; color: var(--color-medium-gray); font-size: 12px;">${this.escapeHtml(tag.description)}</p>` : ''}
                    </div>
                    <div>
                        <button class="btn btn-sm btn-outline" onclick="Tags.editTag(${tag.id})">Editar</button>
                        <button class="btn btn-sm btn-danger" onclick="Tags.deleteTag(${tag.id})">Deletar</button>
                    </div>
                </div>

                ${tag.subtags && tag.subtags.length > 0 ? `
                    <div style="margin-top: 12px; padding-top: 12px; border-top: 1px solid #f0f0f0;">
                        <p style="margin: 0 0 8px 0; font-size: 12px; color: var(--color-medium-gray); font-weight: 600;">Subtags:</p>
                        <div style="display: flex; flex-wrap: wrap; gap: 8px;">
                            ${tag.subtags.map(subtag => `
                                <span style="display: inline-flex; align-items: center; gap: 6px; padding: 4px 12px; background-color: ${subtag.color}20; border: 1px solid ${subtag.color}; border-radius: 20px; font-size: 12px; color: var(--color-dark-navy);">
                                    ${this.escapeHtml(subtag.name)}
                                    <button onclick="Tags.deleteTag(${subtag.id})" style="background: none; border: none; color: var(--color-dark-red); cursor: pointer; padding: 0; font-size: 14px;">×</button>
                                </span>
                            `).join('')}
                        </div>
                    </div>
                ` : ''}
            </div>
        `).join('');
    },

    // Preencher select de tags principais
    populateParentTagSelect() {
        const select = document.getElementById('tag-parent');
        if (!select) return;

        select.innerHTML = '<option value="">Nenhuma (Tag Principal)</option>';
        this.hierarchy.forEach(tag => {
            const option = document.createElement('option');
            option.value = tag.id;
            option.textContent = tag.name;
            select.appendChild(option);
        });
    },

    // Setup de eventos
    setupEventListeners() {
        // Botões de ação
        document.querySelectorAll('[data-modal-trigger]').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                const modalId = btn.getAttribute('data-modal-trigger');
                APP.openModal(modalId);
            });
        });
    },

    // Editar tag
    editTag(id) {
        APP.showNotification('Funcionalidade em desenvolvimento', 'info');
    },

    // Deletar tag
    async deleteTag(id) {
        if (!confirm('Tem certeza que deseja deletar esta tag?')) {
            return;
        }

        try {
            const response = await APP.request(
                `${APP.baseUrl}api/tags/delete?id=${id}`,
                { method: 'DELETE' }
            );

            if (response.success) {
                APP.showNotification('Tag deletada com sucesso!', 'success');
                this.loadTags();
            }
        } catch (error) {
            console.error('Erro ao deletar tag:', error);
            APP.showNotification(error.message || 'Erro ao deletar tag', 'error');
        }
    },

    // Escapar HTML
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
};

// Função para submeter formulário de tag
async function submitTagForm() {
    const form = document.getElementById('tag-form');
    
    if (!APP.validateForm(form)) {
        return;
    }

    const formData = new FormData(form);
    const data = {
        name: formData.get('name'),
        parent_tag_id: formData.get('parent_tag_id') || null,
        description: formData.get('description'),
        color: formData.get('color')
    };

    try {
        const response = await APP.request(
            `${APP.baseUrl}api/tags/create`,
            {
                method: 'POST',
                body: JSON.stringify(data)
            }
        );

        if (response.success) {
            APP.showNotification('Tag criada com sucesso!', 'success');
            APP.closeModal('add-tag-modal');
            form.reset();
            
            // Recarregar tags
            Tags.loadTags();
        }
    } catch (error) {
        console.error('Erro ao criar tag:', error);
        APP.showNotification(error.message || 'Erro ao criar tag', 'error');
    }
}

// Inicializar quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    Tags.init();
});

// Exportar para uso global
window.Tags = Tags;
