/**
 * Finance Manager - Relatórios JavaScript
 */

const Reports = {
    charts: {},
    reportData: {},
    accounts: [],
    categories: [],

    // Inicializar
    async init() {
        try {
            // Carregar dados necessários
            await this.loadAccounts();
            await this.loadCategories();
            
            // Carregar relatório inicial
            await this.loadReport();
            
            // Setup de eventos
            this.setupEventListeners();
        } catch (error) {
            console.error('Erro ao inicializar relatórios:', error);
            APP.showNotification('Erro ao carregar relatórios', 'error');
        }
    },

    // Carregar contas
    async loadAccounts() {
        try {
            const response = await APP.request(
                `${APP.baseUrl}api/accounts/list`,
                { method: 'GET' }
            );

            if (response.success) {
                this.accounts = response.accounts;
                this.populateAccountSelect();
            }
        } catch (error) {
            console.error('Erro ao carregar contas:', error);
        }
    },

    // Carregar categorias
    async loadCategories() {
        try {
            const response = await APP.request(
                `${APP.baseUrl}api/categories/list`,
                { method: 'GET' }
            );

            if (response.success) {
                this.categories = response.categories;
                this.populateCategorySelect();
            }
        } catch (error) {
            console.error('Erro ao carregar categorias:', error);
        }
    },

    // Preencher select de contas
    populateAccountSelect() {
        const select = document.getElementById('filter-account');
        if (!select) return;

        select.innerHTML = '<option value="">Todas</option>';
        this.accounts.forEach(account => {
            const option = document.createElement('option');
            option.value = account.id;
            option.textContent = account.name;
            select.appendChild(option);
        });
    },

    // Preencher select de categorias
    populateCategorySelect() {
        const select = document.getElementById('filter-category');
        if (!select) return;

        select.innerHTML = '<option value="">Todas</option>';
        this.categories.forEach(category => {
            const option = document.createElement('option');
            option.value = category.id;
            option.textContent = category.name;
            select.appendChild(option);
        });
    },

    // Carregar relatório
    async loadReport() {
        try {
            const filters = this.getFilters();
            const queryString = new URLSearchParams(filters).toString();
            
            const response = await APP.request(
                `${APP.baseUrl}api/reports/get?${queryString}`,
                { method: 'GET' }
            );

            if (response.success) {
                this.reportData = response;
                this.updateSummary();
                this.renderTransactions();
                this.renderCharts();
            }
        } catch (error) {
            console.error('Erro ao carregar relatório:', error);
            APP.showNotification('Erro ao carregar relatório', 'error');
        }
    },

    // Obter filtros
    getFilters() {
        return {
            start_date: document.getElementById('filter-start-date')?.value || '',
            end_date: document.getElementById('filter-end-date')?.value || '',
            type: document.getElementById('filter-type')?.value || '',
            account_id: document.getElementById('filter-account')?.value || '',
            category_id: document.getElementById('filter-category')?.value || '',
            limit: 100
        };
    },

    // Atualizar resumo
    updateSummary() {
        const totals = this.reportData.totals || {};
        const income = parseFloat(totals.total_income) || 0;
        const expense = parseFloat(totals.total_expense) || 0;
        const balance = income - expense;

        document.getElementById('total-income').textContent = APP.formatCurrency(income);
        document.getElementById('total-expense').textContent = APP.formatCurrency(expense);
        document.getElementById('total-balance').textContent = APP.formatCurrency(balance);
        document.getElementById('total-transactions').textContent = totals.total_transactions || 0;
    },

    // Renderizar transações
    renderTransactions() {
        const tbody = document.getElementById('transactions-table');
        if (!tbody || !this.reportData.transactions) return;

        if (this.reportData.transactions.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="7" style="text-align: center; padding: 40px;">
                        <p style="color: var(--color-medium-gray);">Nenhuma transação encontrada</p>
                    </td>
                </tr>
            `;
            return;
        }

        tbody.innerHTML = this.reportData.transactions.map(transaction => `
            <tr>
                <td>${APP.formatDate(transaction.transaction_date)}</td>
                <td>${this.escapeHtml(transaction.description || 'Transação')}</td>
                <td>${this.escapeHtml(transaction.category_name || '-')}</td>
                <td>${this.escapeHtml(transaction.account_name || '-')}</td>
                <td>
                    <span class="badge ${this.getBadgeClass(transaction.type)}">
                        ${this.getTypeLabel(transaction.type)}
                    </span>
                </td>
                <td style="text-align: right; font-weight: 600; color: ${this.getAmountColor(transaction.type)};">
                    ${(transaction.type === 'income' ? '+' : '-')} ${APP.formatCurrency(transaction.amount)}
                </td>
                <td>
                    ${transaction.tags ? `<small>${this.escapeHtml(transaction.tags)}</small>` : '-'}
                </td>
            </tr>
        `).join('');
    },

    // Renderizar gráficos
    renderCharts() {
        this.renderCategoryChart();
        this.renderAccountChart();
    },

    // Renderizar gráfico de categorias
    renderCategoryChart() {
        const ctx = document.getElementById('category-chart');
        if (!ctx || !this.reportData.summary_by_category) return;

        const labels = [];
        const data = [];
        const colors = [
            '#2D5016',
            '#1A3A52',
            '#8B0000',
            '#FFA500',
            '#E8E8E8',
            '#666666',
            '#333333',
            '#CCCCCC'
        ];

        this.reportData.summary_by_category.forEach((item, index) => {
            labels.push(item.name || 'Sem categoria');
            data.push(parseFloat(item.total) || 0);
        });

        if (this.charts.category) {
            this.charts.category.destroy();
        }

        this.charts.category = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: labels,
                datasets: [
                    {
                        data: data,
                        backgroundColor: colors.slice(0, data.length),
                        borderColor: '#fff',
                        borderWidth: 2
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: true,
                        position: 'right',
                        labels: {
                            font: {
                                family: "'Segoe UI', Tahoma, Geneva, Verdana, sans-serif",
                                size: 12
                            },
                            color: '#333',
                            padding: 15
                        }
                    }
                }
            }
        });
    },

    // Renderizar gráfico de contas
    renderAccountChart() {
        const ctx = document.getElementById('account-chart');
        if (!ctx || !this.reportData.summary_by_account) return;

        const labels = [];
        const incomeData = [];
        const expenseData = [];

        this.reportData.summary_by_account.forEach(item => {
            labels.push(item.name);
            incomeData.push(parseFloat(item.income) || 0);
            expenseData.push(parseFloat(item.expense) || 0);
        });

        if (this.charts.account) {
            this.charts.account.destroy();
        }

        this.charts.account = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [
                    {
                        label: 'Receitas',
                        data: incomeData,
                        backgroundColor: '#2D5016'
                    },
                    {
                        label: 'Despesas',
                        data: expenseData,
                        backgroundColor: '#8B0000'
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: true,
                        position: 'top',
                        labels: {
                            font: {
                                family: "'Segoe UI', Tahoma, Geneva, Verdana, sans-serif",
                                size: 12
                            },
                            color: '#333',
                            padding: 15
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)'
                        },
                        ticks: {
                            color: '#666',
                            font: {
                                size: 11
                            }
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        },
                        ticks: {
                            color: '#666',
                            font: {
                                size: 11
                            }
                        }
                    }
                }
            }
        });
    },

    // Setup de eventos
    setupEventListeners() {
        // Botão de aplicar filtros
        document.querySelectorAll('button[onclick="Reports.applyFilters()"]').forEach(btn => {
            btn.addEventListener('click', () => this.applyFilters());
        });
    },

    // Aplicar filtros
    async applyFilters() {
        await this.loadReport();
    },

    // Exportar PDF
    async exportPDF() {
        try {
            const filters = this.getFilters();
            const queryString = new URLSearchParams(filters).toString();
            
            // Redirecionar para endpoint de exportação
            window.location.href = `${APP.baseUrl}api/reports/export-pdf?${queryString}`;
            
            APP.showNotification('Relatório sendo gerado...', 'info');
        } catch (error) {
            console.error('Erro ao exportar PDF:', error);
            APP.showNotification('Erro ao exportar PDF', 'error');
        }
    },

    // Obter classe de badge
    getBadgeClass(type) {
        switch (type) {
            case 'income':
                return 'badge-success';
            case 'expense':
                return 'badge-danger';
            case 'transfer':
                return 'badge-info';
            default:
                return 'badge-primary';
        }
    },

    // Obter rótulo de tipo
    getTypeLabel(type) {
        switch (type) {
            case 'income':
                return 'Receita';
            case 'expense':
                return 'Despesa';
            case 'transfer':
                return 'Transferência';
            default:
                return type;
        }
    },

    // Obter cor do valor
    getAmountColor(type) {
        return type === 'income' ? 'var(--color-military-green)' : 'var(--color-dark-red)';
    },

    // Escapar HTML
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
};

// Inicializar quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    Reports.init();
});

// Exportar para uso global
window.Reports = Reports;
