/**
 * Finance Manager - Open Banking JavaScript
 */

const OpenBanking = {
    connections: [],

    // Inicializar
    async init() {
        try {
            // Carregar conexões
            await this.loadConnections();
            
            // Setup de eventos
            this.setupEventListeners();
        } catch (error) {
            console.error('Erro ao inicializar Open Banking:', error);
            APP.showNotification('Erro ao carregar conexões', 'error');
        }
    },

    // Carregar conexões
    async loadConnections() {
        try {
            const response = await APP.request(
                `${APP.baseUrl}api/open-banking/connections`,
                { method: 'GET' }
            );

            if (response.success) {
                this.connections = response.connections;
                this.renderConnections();
            }
        } catch (error) {
            console.error('Erro ao carregar conexões:', error);
            APP.showNotification('Erro ao carregar conexões', 'error');
        }
    },

    // Renderizar conexões
    renderConnections() {
        const list = document.getElementById('connections-list');
        if (!list) return;

        if (this.connections.length === 0) {
            list.innerHTML = `
                <p style="color: var(--color-medium-gray); text-align: center;">
                    Nenhuma conexão estabelecida. Clique em "Conectar Banco" para começar.
                </p>
            `;
            return;
        }

        list.innerHTML = this.connections.map(connection => `
            <div style="margin-bottom: 16px; padding: 16px; border: 1px solid #e0e0e0; border-radius: 8px; border-left: 4px solid var(--color-dark-navy);">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 12px;">
                    <div>
                        <h4 style="margin: 0; color: var(--color-dark-navy);">
                            🏦 ${this.escapeHtml(connection.institution_name)}
                        </h4>
                        <p style="margin: 4px 0 0 0; color: var(--color-medium-gray); font-size: 12px;">
                            Código: ${connection.bank_code} • Conta: ${connection.account_number}
                        </p>
                    </div>
                    <span class="badge ${connection.is_active ? 'badge-success' : 'badge-danger'}">
                        ${connection.is_active ? 'Ativa' : 'Inativa'}
                    </span>
                </div>

                <div style="display: flex; gap: 8px; margin-top: 12px;">
                    <button class="btn btn-sm btn-primary" onclick="OpenBanking.syncTransactions(${connection.id})">
                        🔄 Sincronizar
                    </button>
                    <button class="btn btn-sm btn-outline" onclick="OpenBanking.viewDetails(${connection.id})">
                        Detalhes
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="OpenBanking.deleteConnection(${connection.id})" style="margin-left: auto;">
                        Deletar
                    </button>
                </div>

                ${connection.last_sync_at ? `
                    <p style="margin: 12px 0 0 0; color: var(--color-medium-gray); font-size: 12px;">
                        Última sincronização: ${this.formatDateTime(connection.last_sync_at)}
                    </p>
                ` : ''}
            </div>
        `).join('');
    },

    // Setup de eventos
    setupEventListeners() {
        // Botões de ação
        document.querySelectorAll('[data-modal-trigger]').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                const modalId = btn.getAttribute('data-modal-trigger');
                APP.openModal(modalId);
            });
        });
    },

    // Conectar banco
    async connectBank() {
        const bankSelect = document.getElementById('bank-select');
        const bankCode = bankSelect.value;

        if (!bankCode) {
            APP.showNotification('Selecione um banco', 'warning');
            return;
        }

        try {
            const response = await APP.request(
                `${APP.baseUrl}api/open-banking/auth-url?bank_code=${bankCode}`,
                { method: 'GET' }
            );

            if (response.success) {
                // Redirecionar para autorização
                window.location.href = response.authorization_url;
            }
        } catch (error) {
            console.error('Erro ao conectar banco:', error);
            APP.showNotification('Erro ao conectar banco', 'error');
        }
    },

    // Sincronizar transações
    async syncTransactions(connectionId) {
        try {
            APP.showNotification('Sincronizando transações...', 'info');

            const response = await APP.request(
                `${APP.baseUrl}api/open-banking/sync?connection_id=${connectionId}`,
                {
                    method: 'POST',
                    body: JSON.stringify({
                        start_date: null,
                        end_date: null
                    })
                }
            );

            if (response.success) {
                APP.showNotification(
                    `${response.imported} transações importadas com sucesso!`,
                    'success'
                );
                this.loadConnections();
            }
        } catch (error) {
            console.error('Erro ao sincronizar:', error);
            APP.showNotification('Erro ao sincronizar transações', 'error');
        }
    },

    // Visualizar detalhes
    async viewDetails(connectionId) {
        try {
            const response = await APP.request(
                `${APP.baseUrl}api/open-banking/sync-status?connection_id=${connectionId}`,
                { method: 'GET' }
            );

            if (response.success) {
                const status = response.status;
                const message = `
                    Banco: ${status.institution_name}
                    Status: ${status.is_active ? 'Ativa' : 'Inativa'}
                    Criada em: ${this.formatDateTime(status.created_at)}
                    Última sincronização: ${status.last_sync_at ? this.formatDateTime(status.last_sync_at) : 'Nunca'}
                `;
                APP.showNotification(message, 'info');
            }
        } catch (error) {
            console.error('Erro ao obter detalhes:', error);
            APP.showNotification('Erro ao obter detalhes', 'error');
        }
    },

    // Deletar conexão
    async deleteConnection(connectionId) {
        if (!confirm('Tem certeza que deseja deletar esta conexão?')) {
            return;
        }

        try {
            const response = await APP.request(
                `${APP.baseUrl}api/open-banking/delete?connection_id=${connectionId}`,
                { method: 'DELETE' }
            );

            if (response.success) {
                APP.showNotification('Conexão deletada com sucesso!', 'success');
                this.loadConnections();
            }
        } catch (error) {
            console.error('Erro ao deletar conexão:', error);
            APP.showNotification('Erro ao deletar conexão', 'error');
        }
    },

    // Formatar data e hora
    formatDateTime(dateStr) {
        if (!dateStr) return '-';
        const date = new Date(dateStr);
        return date.toLocaleDateString('pt-BR') + ' ' + date.toLocaleTimeString('pt-BR');
    },

    // Escapar HTML
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
};

// Inicializar quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    OpenBanking.init();
});

// Exportar para uso global
window.OpenBanking = OpenBanking;
