/**
 * Finance Manager - Dashboard JavaScript
 */

const Dashboard = {
    charts: {},
    
    // Inicializar dashboard
    async init() {
        try {
            // Carregar dados do dashboard
            await this.loadDashboardData();
            
            // Carregar dados dos gráficos
            await this.loadChartData();
            
            // Renderizar gráficos
            this.renderCharts();
            
            // Setup de eventos
            this.setupEventListeners();
        } catch (error) {
            console.error('Erro ao inicializar dashboard:', error);
            APP.showNotification('Erro ao carregar dashboard', 'error');
        }
    },

    // Carregar dados do dashboard
    async loadDashboardData() {
        try {
            const response = await APP.request(
                `${APP.baseUrl}api/dashboard/data`,
                { method: 'GET' }
            );

            if (response.success) {
                this.dashboardData = response;
            }
        } catch (error) {
            console.error('Erro ao carregar dados do dashboard:', error);
        }
    },

    // Carregar dados dos gráficos
    async loadChartData() {
        try {
            const response = await APP.request(
                `${APP.baseUrl}api/dashboard/chart-data`,
                { method: 'GET' }
            );

            if (response.success) {
                this.chartData = response;
            }
        } catch (error) {
            console.error('Erro ao carregar dados dos gráficos:', error);
        }
    },

    // Renderizar gráficos
    renderCharts() {
        if (!this.chartData) return;

        // Gráfico de Receitas vs Despesas
        this.renderIncomeExpenseChart();

        // Gráfico de Distribuição por Categoria
        this.renderCategoryChart();
    },

    // Renderizar gráfico de receitas vs despesas
    renderIncomeExpenseChart() {
        const ctx = document.getElementById('income-expense-chart');
        if (!ctx) return;

        const labels = [];
        const incomeData = [];
        const expenseData = [];

        // Processar dados
        this.chartData.chart_data.forEach(item => {
            labels.push(this.formatChartDate(item.date));
            incomeData.push(parseFloat(item.income) || 0);
            expenseData.push(parseFloat(item.expense) || 0);
        });

        this.charts.incomeExpense = new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [
                    {
                        label: 'Receitas',
                        data: incomeData,
                        borderColor: '#2D5016',
                        backgroundColor: 'rgba(45, 80, 22, 0.1)',
                        tension: 0.4,
                        fill: true,
                        pointRadius: 4,
                        pointBackgroundColor: '#2D5016',
                        pointBorderColor: '#fff',
                        pointBorderWidth: 2
                    },
                    {
                        label: 'Despesas',
                        data: expenseData,
                        borderColor: '#8B0000',
                        backgroundColor: 'rgba(139, 0, 0, 0.1)',
                        tension: 0.4,
                        fill: true,
                        pointRadius: 4,
                        pointBackgroundColor: '#8B0000',
                        pointBorderColor: '#fff',
                        pointBorderWidth: 2
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: true,
                        position: 'top',
                        labels: {
                            font: {
                                family: "'Segoe UI', Tahoma, Geneva, Verdana, sans-serif",
                                size: 12
                            },
                            color: '#333',
                            padding: 15
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)'
                        },
                        ticks: {
                            color: '#666',
                            font: {
                                size: 11
                            }
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        },
                        ticks: {
                            color: '#666',
                            font: {
                                size: 11
                            }
                        }
                    }
                }
            }
        });
    },

    // Renderizar gráfico de categorias
    renderCategoryChart() {
        const ctx = document.getElementById('category-chart');
        if (!ctx) return;

        const labels = [];
        const data = [];
        const colors = [
            '#2D5016',
            '#1A3A52',
            '#8B0000',
            '#FFA500',
            '#E8E8E8',
            '#666666',
            '#333333',
            '#CCCCCC'
        ];

        // Processar dados
        this.chartData.category_data.forEach((item, index) => {
            labels.push(item.category || 'Sem categoria');
            data.push(parseFloat(item.total) || 0);
        });

        this.charts.category = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: labels,
                datasets: [
                    {
                        data: data,
                        backgroundColor: colors.slice(0, data.length),
                        borderColor: '#fff',
                        borderWidth: 2
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: true,
                        position: 'right',
                        labels: {
                            font: {
                                family: "'Segoe UI', Tahoma, Geneva, Verdana, sans-serif",
                                size: 12
                            },
                            color: '#333',
                            padding: 15
                        }
                    }
                }
            }
        });
    },

    // Setup de eventos
    setupEventListeners() {
        // Botões de ação
        document.querySelectorAll('[data-modal-trigger]').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                const modalId = btn.getAttribute('data-modal-trigger');
                APP.openModal(modalId);
            });
        });
    },

    // Formatar data para gráfico
    formatChartDate(dateStr) {
        const date = new Date(dateStr);
        return date.toLocaleDateString('pt-BR', { month: 'short', day: 'numeric' });
    }
};

// Função para submeter formulário de receita
async function submitIncomeForm() {
    const form = document.getElementById('income-form');
    
    if (!APP.validateForm(form)) {
        return;
    }

    const formData = new FormData(form);
    const data = {
        type: 'income',
        description: formData.get('description'),
        amount: parseFloat(formData.get('amount')),
        account_id: parseInt(formData.get('account_id')),
        transaction_date: formData.get('transaction_date')
    };

    try {
        const response = await APP.request(
            `${APP.baseUrl}api/transactions/create`,
            {
                method: 'POST',
                body: JSON.stringify(data)
            }
        );

        if (response.success) {
            APP.showNotification('Receita registrada com sucesso!', 'success');
            APP.closeModal('add-income-modal');
            form.reset();
            
            // Recarregar dados
            setTimeout(() => {
                location.reload();
            }, 1500);
        }
    } catch (error) {
        console.error('Erro ao registrar receita:', error);
        APP.showNotification(error.message || 'Erro ao registrar receita', 'error');
    }
}

// Função para submeter formulário de despesa
async function submitExpenseForm() {
    const form = document.getElementById('expense-form');
    
    if (!APP.validateForm(form)) {
        return;
    }

    const formData = new FormData(form);
    const data = {
        type: 'expense',
        description: formData.get('description'),
        amount: parseFloat(formData.get('amount')),
        account_id: parseInt(formData.get('account_id')),
        transaction_date: formData.get('transaction_date')
    };

    try {
        const response = await APP.request(
            `${APP.baseUrl}api/transactions/create`,
            {
                method: 'POST',
                body: JSON.stringify(data)
            }
        );

        if (response.success) {
            APP.showNotification('Despesa registrada com sucesso!', 'success');
            APP.closeModal('add-expense-modal');
            form.reset();
            
            // Recarregar dados
            setTimeout(() => {
                location.reload();
            }, 1500);
        }
    } catch (error) {
        console.error('Erro ao registrar despesa:', error);
        APP.showNotification(error.message || 'Erro ao registrar despesa', 'error');
    }
}

// Inicializar quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    Dashboard.init();
});

// Exportar para uso global
window.Dashboard = Dashboard;
