/**
 * Finance Manager - Categorias JavaScript
 */

const Categories = {
    categories: [],
    filteredCategories: [],

    // Inicializar
    async init() {
        try {
            // Carregar categorias
            await this.loadCategories();
            
            // Setup de eventos
            this.setupEventListeners();
        } catch (error) {
            console.error('Erro ao inicializar categorias:', error);
            APP.showNotification('Erro ao carregar categorias', 'error');
        }
    },

    // Carregar categorias
    async loadCategories() {
        try {
            const response = await APP.request(
                `${APP.baseUrl}api/categories/list`,
                { method: 'GET' }
            );

            if (response.success) {
                this.categories = response.categories;
                this.filteredCategories = this.categories;
                this.renderCategories();
            }
        } catch (error) {
            console.error('Erro ao carregar categorias:', error);
            APP.showNotification('Erro ao carregar categorias', 'error');
        }
    },

    // Renderizar categorias
    renderCategories() {
        const grid = document.getElementById('categories-grid');
        if (!grid) return;

        if (this.filteredCategories.length === 0) {
            grid.innerHTML = `
                <div style="text-align: center; width: 100%; padding: 40px;">
                    <p style="color: var(--color-medium-gray);">Nenhuma categoria encontrada</p>
                </div>
            `;
            return;
        }

        // Agrupar por tipo
        const grouped = {};
        this.filteredCategories.forEach(cat => {
            if (!grouped[cat.type]) {
                grouped[cat.type] = [];
            }
            grouped[cat.type].push(cat);
        });

        let html = '';

        // Renderizar receitas
        if (grouped['income']) {
            html += `
                <div style="width: 100%; margin-bottom: 24px;">
                    <h3 style="color: var(--color-military-green); margin-bottom: 16px;">Receitas</h3>
                    <div class="row">
                        ${grouped['income'].map(cat => this.renderCategoryCard(cat)).join('')}
                    </div>
                </div>
            `;
        }

        // Renderizar despesas
        if (grouped['expense']) {
            html += `
                <div style="width: 100%; margin-bottom: 24px;">
                    <h3 style="color: var(--color-dark-red); margin-bottom: 16px;">Despesas</h3>
                    <div class="row">
                        ${grouped['expense'].map(cat => this.renderCategoryCard(cat)).join('')}
                    </div>
                </div>
            `;
        }

        grid.innerHTML = html;
    },

    // Renderizar card de categoria
    renderCategoryCard(category) {
        return `
            <div class="col">
                <div class="card" style="border-left: 4px solid ${category.color};">
                    <div class="card-header">
                        <div>
                            <h4 style="margin: 0; color: var(--color-dark-navy);">
                                ${this.escapeHtml(category.name)}
                            </h4>
                            ${category.description ? `<p style="margin: 4px 0 0 0; color: var(--color-medium-gray); font-size: 12px;">${this.escapeHtml(category.description)}</p>` : ''}
                        </div>
                    </div>
                    <div class="card-body">
                        <div style="display: flex; gap: 8px; margin-bottom: 12px;">
                            <span class="badge ${category.type === 'income' ? 'badge-success' : 'badge-danger'}">
                                ${category.type === 'income' ? 'Receita' : 'Despesa'}
                            </span>
                            <span class="badge ${category.is_active ? 'badge-success' : 'badge-danger'}">
                                ${category.is_active ? 'Ativa' : 'Inativa'}
                            </span>
                        </div>
                    </div>
                    <div class="card-footer">
                        <button class="btn btn-sm btn-outline" onclick="Categories.editCategory(${category.id})">
                            Editar
                        </button>
                        <button class="btn btn-sm btn-danger" onclick="Categories.deleteCategory(${category.id})" style="margin-left: auto;">
                            Deletar
                        </button>
                    </div>
                </div>
            </div>
        `;
    },

    // Setup de eventos
    setupEventListeners() {
        // Botões de ação
        document.querySelectorAll('[data-modal-trigger]').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                const modalId = btn.getAttribute('data-modal-trigger');
                APP.openModal(modalId);
            });
        });
    },

    // Filtrar por tipo
    filterByType(type) {
        if (type === '') {
            this.filteredCategories = this.categories;
        } else {
            this.filteredCategories = this.categories.filter(cat => cat.type === type);
        }
        this.renderCategories();
    },

    // Editar categoria
    editCategory(id) {
        APP.showNotification('Funcionalidade em desenvolvimento', 'info');
    },

    // Deletar categoria
    async deleteCategory(id) {
        if (!confirm('Tem certeza que deseja deletar esta categoria?')) {
            return;
        }

        try {
            const response = await APP.request(
                `${APP.baseUrl}api/categories/delete?id=${id}`,
                { method: 'DELETE' }
            );

            if (response.success) {
                APP.showNotification('Categoria deletada com sucesso!', 'success');
                this.loadCategories();
            }
        } catch (error) {
            console.error('Erro ao deletar categoria:', error);
            APP.showNotification(error.message || 'Erro ao deletar categoria', 'error');
        }
    },

    // Escapar HTML
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
};

// Função para submeter formulário de categoria
async function submitCategoryForm() {
    const form = document.getElementById('category-form');
    
    if (!APP.validateForm(form)) {
        return;
    }

    const formData = new FormData(form);
    const data = {
        name: formData.get('name'),
        type: formData.get('type'),
        description: formData.get('description'),
        color: formData.get('color')
    };

    try {
        const response = await APP.request(
            `${APP.baseUrl}api/categories/create`,
            {
                method: 'POST',
                body: JSON.stringify(data)
            }
        );

        if (response.success) {
            APP.showNotification('Categoria criada com sucesso!', 'success');
            APP.closeModal('add-category-modal');
            form.reset();
            
            // Recarregar categorias
            Categories.loadCategories();
        }
    } catch (error) {
        console.error('Erro ao criar categoria:', error);
        APP.showNotification(error.message || 'Erro ao criar categoria', 'error');
    }
}

// Inicializar quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    Categories.init();
});

// Exportar para uso global
window.Categories = Categories;
