/**
 * Finance Manager - Autenticação JavaScript
 */

const Auth = {
    // Inicializar
    init() {
        this.setupTabs();
        this.setupForms();
        this.setupSocialLogin();
    },

    // Setup de abas
    setupTabs() {
        const tabs = document.querySelectorAll('.auth-tab');
        tabs.forEach(tab => {
            tab.addEventListener('click', () => {
                this.switchTab(tab.getAttribute('data-tab'));
            });
        });
    },

    // Trocar aba
    switchTab(tabName) {
        // Remover classe active de todas as abas e conteúdos
        document.querySelectorAll('.auth-tab').forEach(tab => {
            tab.classList.remove('active');
        });
        document.querySelectorAll('.auth-tab-content').forEach(content => {
            content.classList.remove('active');
        });

        // Adicionar classe active à aba selecionada
        document.querySelector(`[data-tab="${tabName}"]`).classList.add('active');
        document.getElementById(`${tabName}-tab`).classList.add('active');

        // Limpar formulário
        const form = document.getElementById(`${tabName}-form`);
        if (form) {
            form.reset();
            form.querySelectorAll('.form-error').forEach(error => error.remove());
            form.querySelectorAll('.is-invalid').forEach(input => {
                input.classList.remove('is-invalid');
            });
        }
    },

    // Setup de formulários
    setupForms() {
        const loginForm = document.getElementById('login-form');
        const registerForm = document.getElementById('register-form');

        if (loginForm) {
            loginForm.addEventListener('submit', (e) => this.handleLogin(e));
        }

        if (registerForm) {
            registerForm.addEventListener('submit', (e) => this.handleRegister(e));
        }
    },

    // Lidar com login
    async handleLogin(e) {
        e.preventDefault();

        const form = e.target;
        const email = form.querySelector('[name="email"]').value;
        const password = form.querySelector('[name="password"]').value;

        // Validar
        if (!email || !password) {
            APP.showNotification('Email e senha são obrigatórios', 'error');
            return;
        }

        try {
            // Desabilitar botão
            const submitBtn = form.querySelector('button[type="submit"]');
            submitBtn.disabled = true;
            submitBtn.textContent = 'Entrando...';

            const response = await APP.request(
                `${APP.baseUrl}auth/login`,
                {
                    method: 'POST',
                    body: JSON.stringify({ email, password })
                }
            );

            if (response.success) {
                APP.showNotification('Login realizado com sucesso!', 'success');
                setTimeout(() => {
                    window.location.href = `${APP.baseUrl}dashboard`;
                }, 1500);
            }
        } catch (error) {
            console.error('Erro ao fazer login:', error);
            APP.showNotification(error.message || 'Erro ao fazer login', 'error');
            
            // Reabilitar botão
            const submitBtn = form.querySelector('button[type="submit"]');
            submitBtn.disabled = false;
            submitBtn.textContent = 'Entrar';
        }
    },

    // Lidar com registro
    async handleRegister(e) {
        e.preventDefault();

        const form = e.target;
        const firstName = form.querySelector('[name="first_name"]').value;
        const lastName = form.querySelector('[name="last_name"]').value;
        const email = form.querySelector('[name="email"]').value;
        const password = form.querySelector('[name="password"]').value;
        const passwordConfirm = form.querySelector('[name="password_confirm"]').value;
        const terms = form.querySelector('[name="terms"]').checked;

        // Validar
        if (!firstName || !email || !password || !passwordConfirm) {
            APP.showNotification('Preencha todos os campos obrigatórios', 'error');
            return;
        }

        if (password !== passwordConfirm) {
            APP.showNotification('As senhas não conferem', 'error');
            return;
        }

        if (password.length < 8) {
            APP.showNotification('A senha deve ter no mínimo 8 caracteres', 'error');
            return;
        }

        if (!terms) {
            APP.showNotification('Você deve concordar com os Termos de Serviço', 'error');
            return;
        }

        try {
            // Desabilitar botão
            const submitBtn = form.querySelector('button[type="submit"]');
            submitBtn.disabled = true;
            submitBtn.textContent = 'Criando conta...';

            const response = await APP.request(
                `${APP.baseUrl}auth/register`,
                {
                    method: 'POST',
                    body: JSON.stringify({
                        first_name: firstName,
                        last_name: lastName,
                        email,
                        password,
                        password_confirm: passwordConfirm
                    })
                }
            );

            if (response.success) {
                APP.showNotification('Conta criada com sucesso! Redirecionando...', 'success');
                setTimeout(() => {
                    window.location.href = `${APP.baseUrl}dashboard`;
                }, 1500);
            }
        } catch (error) {
            console.error('Erro ao criar conta:', error);
            APP.showNotification(error.message || 'Erro ao criar conta', 'error');
            
            // Reabilitar botão
            const submitBtn = form.querySelector('button[type="submit"]');
            submitBtn.disabled = false;
            submitBtn.textContent = 'Criar Conta';
        }
    },

    // Setup de login social
    setupSocialLogin() {
        const socialButtons = document.querySelectorAll('.btn-social');
        socialButtons.forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                const provider = btn.getAttribute('data-provider');
                this.handleSocialLogin(provider);
            });
        });
    },

    // Lidar com login social
    handleSocialLogin(provider) {
        if (provider === 'google') {
            // Redirecionar para Google OAuth
            const clientId = 'SEU_GOOGLE_CLIENT_ID';
            const redirectUri = `${APP.baseUrl}auth/google-callback`;
            const scope = 'openid profile email';
            const responseType = 'code';

            const googleAuthUrl = `https://accounts.google.com/o/oauth2/v2/auth?` +
                `client_id=${clientId}&` +
                `redirect_uri=${encodeURIComponent(redirectUri)}&` +
                `response_type=${responseType}&` +
                `scope=${encodeURIComponent(scope)}`;

            window.location.href = googleAuthUrl;
        } else if (provider === 'apple') {
            // Redirecionar para Apple Sign In
            const clientId = 'SEU_APPLE_CLIENT_ID';
            const redirectUri = `${APP.baseUrl}auth/apple-callback`;
            const teamId = 'SEU_APPLE_TEAM_ID';
            const keyId = 'SEU_APPLE_KEY_ID';

            const appleAuthUrl = `https://appleid.apple.com/auth/authorize?` +
                `client_id=${clientId}&` +
                `redirect_uri=${encodeURIComponent(redirectUri)}&` +
                `response_type=code&` +
                `response_mode=form_post&` +
                `scope=name email`;

            window.location.href = appleAuthUrl;
        }
    }
};

// Inicializar quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    Auth.init();
});

// Exportar para uso global
window.Auth = Auth;
