/**
 * Finance Manager - JavaScript Principal
 */

// Configurações globais
const APP = {
    baseUrl: document.querySelector('meta[name="base-url"]')?.content || '/',
    userId: document.querySelector('meta[name="user-id"]')?.content || null,
    currency: document.querySelector('meta[name="currency"]')?.content || 'BRL',
    
    // Inicializar aplicação
    init() {
        this.setupEventListeners();
        this.setupNavigation();
        this.setupModals();
        this.setupFormValidation();
    },

    // Setup de listeners de eventos
    setupEventListeners() {
        // Toggle do menu mobile
        const navbarToggle = document.querySelector('.navbar-toggle');
        if (navbarToggle) {
            navbarToggle.addEventListener('click', () => {
                const navbarMenu = document.querySelector('.navbar-menu');
                navbarMenu?.classList.toggle('active');
            });
        }

        // Toggle do sidebar mobile
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                const sidebar = document.querySelector('.sidebar');
                sidebar?.classList.toggle('active');
            });
        }

        // Fechar menu ao clicar em um link
        document.querySelectorAll('.navbar-menu a').forEach(link => {
            link.addEventListener('click', () => {
                document.querySelector('.navbar-menu')?.classList.remove('active');
            });
        });

        // Fechar sidebar ao clicar em um link
        document.querySelectorAll('.sidebar-menu a').forEach(link => {
            link.addEventListener('click', () => {
                if (window.innerWidth <= 768) {
                    document.querySelector('.sidebar')?.classList.remove('active');
                }
            });
        });

        // Fechar modais ao clicar fora
        document.querySelectorAll('.modal').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    this.closeModal(modal.id);
                }
            });
        });
    },

    // Setup de navegação
    setupNavigation() {
        const currentPath = window.location.pathname;
        document.querySelectorAll('.navbar-menu a, .sidebar-menu a').forEach(link => {
            const href = link.getAttribute('href');
            if (href && currentPath.includes(href)) {
                link.classList.add('active');
            }
        });
    },

    // Setup de modais
    setupModals() {
        // Abrir modal
        document.querySelectorAll('[data-modal-trigger]').forEach(trigger => {
            trigger.addEventListener('click', (e) => {
                e.preventDefault();
                const modalId = trigger.getAttribute('data-modal-trigger');
                this.openModal(modalId);
            });
        });

        // Fechar modal
        document.querySelectorAll('.modal-close').forEach(closeBtn => {
            closeBtn.addEventListener('click', () => {
                const modal = closeBtn.closest('.modal');
                if (modal) {
                    this.closeModal(modal.id);
                }
            });
        });
    },

    // Setup de validação de formulários
    setupFormValidation() {
        document.querySelectorAll('form[data-validate]').forEach(form => {
            form.addEventListener('submit', (e) => {
                if (!this.validateForm(form)) {
                    e.preventDefault();
                }
            });
        });
    },

    // Abrir modal
    openModal(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.add('active');
            document.body.style.overflow = 'hidden';
        }
    },

    // Fechar modal
    closeModal(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.remove('active');
            document.body.style.overflow = 'auto';
        }
    },

    // Validar formulário
    validateForm(form) {
        let isValid = true;
        const inputs = form.querySelectorAll('input, textarea, select');

        inputs.forEach(input => {
            const error = this.validateInput(input);
            if (error) {
                isValid = false;
                this.showFieldError(input, error);
            } else {
                this.clearFieldError(input);
            }
        });

        return isValid;
    },

    // Validar campo individual
    validateInput(input) {
        const value = input.value.trim();
        const type = input.type;
        const required = input.hasAttribute('required');

        if (required && !value) {
            return 'Este campo é obrigatório';
        }

        if (value) {
            if (type === 'email' && !this.isValidEmail(value)) {
                return 'Email inválido';
            }

            if (type === 'number') {
                if (isNaN(value)) {
                    return 'Deve ser um número';
                }
                if (input.min && parseFloat(value) < parseFloat(input.min)) {
                    return `Deve ser maior que ${input.min}`;
                }
                if (input.max && parseFloat(value) > parseFloat(input.max)) {
                    return `Deve ser menor que ${input.max}`;
                }
            }

            if (input.minLength && value.length < input.minLength) {
                return `Mínimo de ${input.minLength} caracteres`;
            }

            if (input.maxLength && value.length > input.maxLength) {
                return `Máximo de ${input.maxLength} caracteres`;
            }
        }

        return null;
    },

    // Validar email
    isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    },

    // Mostrar erro de campo
    showFieldError(input, error) {
        input.classList.add('is-invalid');
        let errorElement = input.nextElementSibling;

        if (!errorElement || !errorElement.classList.contains('form-error')) {
            errorElement = document.createElement('div');
            errorElement.className = 'form-error';
            input.parentNode.insertBefore(errorElement, input.nextSibling);
        }

        errorElement.textContent = error;
    },

    // Limpar erro de campo
    clearFieldError(input) {
        input.classList.remove('is-invalid');
        const errorElement = input.nextElementSibling;
        if (errorElement && errorElement.classList.contains('form-error')) {
            errorElement.remove();
        }
    },

    // Fazer requisição AJAX
    async request(url, options = {}) {
        const defaultOptions = {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        };

        const finalOptions = { ...defaultOptions, ...options };

        try {
            const response = await fetch(url, finalOptions);
            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.error || 'Erro na requisição');
            }

            return data;
        } catch (error) {
            console.error('Erro na requisição:', error);
            this.showNotification(error.message, 'error');
            throw error;
        }
    },

    // Mostrar notificação
    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `alert alert-${type}`;
        notification.innerHTML = `
            <span>${message}</span>
            <button class="close-notification" style="background: none; border: none; cursor: pointer; color: inherit;">×</button>
        `;

        const container = document.querySelector('.notification-container') || document.body;
        container.appendChild(notification);

        notification.querySelector('.close-notification').addEventListener('click', () => {
            notification.remove();
        });

        setTimeout(() => {
            notification.remove();
        }, 5000);
    },

    // Formatar moeda
    formatCurrency(value, currency = this.currency) {
        const formatter = new Intl.NumberFormat('pt-BR', {
            style: 'currency',
            currency: currency
        });
        return formatter.format(value);
    },

    // Formatar data
    formatDate(date, format = 'dd/MM/yyyy') {
        const d = new Date(date);
        const day = String(d.getDate()).padStart(2, '0');
        const month = String(d.getMonth() + 1).padStart(2, '0');
        const year = d.getFullYear();

        return format
            .replace('dd', day)
            .replace('MM', month)
            .replace('yyyy', year);
    },

    // Obter usuário atual
    async getCurrentUser() {
        try {
            const response = await this.request(`${this.baseUrl}api/auth/current-user`);
            return response.user;
        } catch (error) {
            console.error('Erro ao obter usuário:', error);
            return null;
        }
    },

    // Fazer logout
    async logout() {
        try {
            await this.request(`${this.baseUrl}api/auth/logout`, { method: 'POST' });
            window.location.href = `${this.baseUrl}auth/login`;
        } catch (error) {
            console.error('Erro ao fazer logout:', error);
        }
    }
};

// Inicializar quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    APP.init();
});

// Exportar para uso global
window.APP = APP;
