/**
 * Finance Manager - Contas JavaScript
 */

const Accounts = {
    accounts: [],
    totalBalance: 0,

    // Inicializar
    async init() {
        try {
            // Carregar contas
            await this.loadAccounts();
            
            // Setup de eventos
            this.setupEventListeners();
        } catch (error) {
            console.error('Erro ao inicializar contas:', error);
            APP.showNotification('Erro ao carregar contas', 'error');
        }
    },

    // Carregar contas
    async loadAccounts() {
        try {
            const response = await APP.request(
                `${APP.baseUrl}api/accounts/list`,
                { method: 'GET' }
            );

            if (response.success) {
                this.accounts = response.accounts;
                this.renderAccounts();
                this.updateSummary();
            }
        } catch (error) {
            console.error('Erro ao carregar contas:', error);
            APP.showNotification('Erro ao carregar contas', 'error');
        }
    },

    // Renderizar contas
    renderAccounts() {
        const grid = document.getElementById('accounts-grid');
        if (!grid) return;

        if (this.accounts.length === 0) {
            grid.innerHTML = `
                <div style="text-align: center; width: 100%; padding: 40px;">
                    <p style="color: var(--color-medium-gray);">Nenhuma conta cadastrada</p>
                    <button class="btn btn-primary" data-modal-trigger="add-account-modal" style="margin-top: 16px;">
                        ➕ Criar Primeira Conta
                    </button>
                </div>
            `;
            return;
        }

        grid.innerHTML = this.accounts.map(account => `
            <div class="col">
                <div class="card" style="border-left: 4px solid ${account.color}; cursor: pointer;" onclick="Accounts.selectAccount(${account.id})">
                    <div class="card-header">
                        <div>
                            <h4 style="margin: 0; color: var(--color-dark-navy);">${this.escapeHtml(account.name)}</h4>
                            <p style="margin: 4px 0 0 0; color: var(--color-medium-gray); font-size: 12px;">
                                ${this.getAccountTypeLabel(account.account_type)}
                                ${account.bank_name ? ' • ' + this.escapeHtml(account.bank_name) : ''}
                            </p>
                        </div>
                        <button class="btn btn-sm btn-outline" onclick="Accounts.editAccount(event, ${account.id})">
                            Editar
                        </button>
                    </div>
                    <div class="card-body">
                        <p style="color: var(--color-medium-gray); margin: 0 0 8px 0; font-size: 12px;">Saldo</p>
                        <p style="font-size: 24px; font-weight: 700; color: ${account.balance >= 0 ? 'var(--color-military-green)' : 'var(--color-dark-red)'}; margin: 0;">
                            ${APP.formatCurrency(account.balance)}
                        </p>
                    </div>
                    <div class="card-footer">
                        <span class="badge ${account.is_active ? 'badge-success' : 'badge-danger'}">
                            ${account.is_active ? 'Ativa' : 'Inativa'}
                        </span>
                        <button class="btn btn-sm btn-danger" onclick="Accounts.deleteAccount(event, ${account.id})" style="margin-left: auto;">
                            Deletar
                        </button>
                    </div>
                </div>
            </div>
        `).join('');
    },

    // Atualizar resumo
    updateSummary() {
        const totalBalance = this.accounts.reduce((sum, account) => sum + parseFloat(account.balance), 0);
        const activeAccounts = this.accounts.filter(a => a.is_active).length;

        document.getElementById('total-balance').textContent = APP.formatCurrency(totalBalance);
        document.getElementById('active-accounts').textContent = activeAccounts;
    },

    // Setup de eventos
    setupEventListeners() {
        // Botões de ação
        document.querySelectorAll('[data-modal-trigger]').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                const modalId = btn.getAttribute('data-modal-trigger');
                APP.openModal(modalId);
            });
        });
    },

    // Obter rótulo do tipo de conta
    getAccountTypeLabel(type) {
        const labels = {
            'checking': 'Conta Corrente',
            'savings': 'Poupança',
            'investment': 'Investimento',
            'credit': 'Crédito',
            'other': 'Outro'
        };
        return labels[type] || type;
    },

    // Selecionar conta
    selectAccount(id) {
        APP.showNotification('Funcionalidade em desenvolvimento', 'info');
    },

    // Editar conta
    editAccount(event, id) {
        event.stopPropagation();
        APP.showNotification('Funcionalidade em desenvolvimento', 'info');
    },

    // Deletar conta
    async deleteAccount(event, id) {
        event.stopPropagation();
        
        if (!confirm('Tem certeza que deseja deletar esta conta?')) {
            return;
        }

        try {
            const response = await APP.request(
                `${APP.baseUrl}api/accounts/delete?id=${id}`,
                { method: 'DELETE' }
            );

            if (response.success) {
                APP.showNotification('Conta deletada com sucesso!', 'success');
                this.loadAccounts();
            }
        } catch (error) {
            console.error('Erro ao deletar conta:', error);
            APP.showNotification(error.message || 'Erro ao deletar conta', 'error');
        }
    },

    // Escapar HTML
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
};

// Função para submeter formulário de conta
async function submitAccountForm() {
    const form = document.getElementById('account-form');
    
    if (!APP.validateForm(form)) {
        return;
    }

    const formData = new FormData(form);
    const data = {
        name: formData.get('name'),
        account_type: formData.get('account_type'),
        bank_name: formData.get('bank_name'),
        balance: parseFloat(formData.get('balance')) || 0
    };

    try {
        const response = await APP.request(
            `${APP.baseUrl}api/accounts/create`,
            {
                method: 'POST',
                body: JSON.stringify(data)
            }
        );

        if (response.success) {
            APP.showNotification('Conta criada com sucesso!', 'success');
            APP.closeModal('add-account-modal');
            form.reset();
            
            // Recarregar contas
            Accounts.loadAccounts();
        }
    } catch (error) {
        console.error('Erro ao criar conta:', error);
        APP.showNotification(error.message || 'Erro ao criar conta', 'error');
    }
}

// Inicializar quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    Accounts.init();
});

// Exportar para uso global
window.Accounts = Accounts;
