<?php
/**
 * Finance Manager - Script de Instalação Automática
 * 
 * Este script configura automaticamente o banco de dados e o arquivo .env
 * Acesse: https://seu-dominio.com/install.php
 */

session_start();

// Verificar se já foi instalado
if (file_exists(__DIR__ . '/.installed')) {
    die('A aplicação já foi instalada. Delete o arquivo .installed para reinstalar.');
}

$step = $_GET['step'] ?? 1;
$errors = [];
$success = [];

// Função para testar conexão com banco de dados
function testDatabaseConnection($host, $user, $password, $database = null) {
    try {
        $mysqli = new mysqli($host, $user, $password, $database);
        if ($mysqli->connect_error) {
            return ['success' => false, 'error' => $mysqli->connect_error];
        }
        $mysqli->close();
        return ['success' => true];
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

// Função para criar banco de dados
function createDatabase($host, $user, $password, $database) {
    try {
        $mysqli = new mysqli($host, $user, $password);
        if ($mysqli->connect_error) {
            return ['success' => false, 'error' => $mysqli->connect_error];
        }
        
        $sql = "CREATE DATABASE IF NOT EXISTS `$database` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci";
        if (!$mysqli->query($sql)) {
            return ['success' => false, 'error' => $mysqli->error];
        }
        
        $mysqli->close();
        return ['success' => true];
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

// Função para importar schema
function importSchema($host, $user, $password, $database) {
    try {
        $mysqli = new mysqli($host, $user, $password, $database);
        if ($mysqli->connect_error) {
            return ['success' => false, 'error' => $mysqli->connect_error];
        }
        
        $schemaFile = __DIR__ . '/database/schema.sql';
        if (!file_exists($schemaFile)) {
            return ['success' => false, 'error' => 'Arquivo schema.sql não encontrado'];
        }
        
        $sql = file_get_contents($schemaFile);
        
        // Executar múltiplas queries
        if ($mysqli->multi_query($sql)) {
            while ($mysqli->more_results()) {
                $mysqli->next_result();
            }
            $mysqli->close();
            return ['success' => true];
        } else {
            return ['success' => false, 'error' => $mysqli->error];
        }
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

// Função para salvar .env
function saveEnvFile($data) {
    try {
        $envContent = "; Finance Manager - Configuração de Ambiente\n\n";
        $envContent .= "; Banco de Dados\n";
        $envContent .= "DB_HOST=" . $data['db_host'] . "\n";
        $envContent .= "DB_PORT=" . $data['db_port'] . "\n";
        $envContent .= "DB_NAME=" . $data['db_name'] . "\n";
        $envContent .= "DB_USER=" . $data['db_user'] . "\n";
        $envContent .= "DB_PASSWORD=" . $data['db_password'] . "\n\n";
        
        $envContent .= "; Aplicação\n";
        $envContent .= "APP_NAME=Finance Manager\n";
        $envContent .= "APP_URL=" . $data['app_url'] . "\n";
        $envContent .= "APP_ENV=production\n";
        $envContent .= "APP_DEBUG=false\n\n";
        
        $envContent .= "; Segurança\n";
        $envContent .= "SESSION_SECRET=" . bin2hex(random_bytes(32)) . "\n";
        $envContent .= "CSRF_TOKEN_EXPIRY=3600\n\n";
        
        $envContent .= "; Google OAuth (opcional)\n";
        $envContent .= "GOOGLE_CLIENT_ID=\n";
        $envContent .= "GOOGLE_CLIENT_SECRET=\n";
        $envContent .= "GOOGLE_REDIRECT_URI=" . $data['app_url'] . "/auth/google/callback\n\n";
        
        $envContent .= "; Apple OAuth (opcional)\n";
        $envContent .= "APPLE_CLIENT_ID=\n";
        $envContent .= "APPLE_TEAM_ID=\n";
        $envContent .= "APPLE_KEY_ID=\n";
        $envContent .= "APPLE_KEY_PATH=/path/to/apple-key.p8\n";
        $envContent .= "APPLE_REDIRECT_URI=" . $data['app_url'] . "/auth/apple/callback\n\n";
        
        $envContent .= "; Open Banking (opcional)\n";
        $envContent .= "OPEN_BANKING_API_KEY=\n";
        $envContent .= "OPEN_BANKING_API_URL=https://api.openbanking.example.com\n";
        $envContent .= "OPEN_BANKING_CLIENT_ID=\n";
        $envContent .= "OPEN_BANKING_CLIENT_SECRET=\n\n";
        
        $envContent .= "; Email (opcional)\n";
        $envContent .= "MAIL_HOST=smtp.mailtrap.io\n";
        $envContent .= "MAIL_PORT=587\n";
        $envContent .= "MAIL_USERNAME=\n";
        $envContent .= "MAIL_PASSWORD=\n";
        $envContent .= "MAIL_FROM=noreply@" . str_replace(['https://', 'http://'], '', $data['app_url']) . "\n\n";
        
        $envContent .= "; Timezone\n";
        $envContent .= "TIMEZONE=America/Sao_Paulo\n";
        
        $envFile = __DIR__ . '/.env';
        if (file_put_contents($envFile, $envContent) === false) {
            return ['success' => false, 'error' => 'Não foi possível salvar o arquivo .env'];
        }
        
        return ['success' => true];
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

// Processar formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($step == 1) {
        // Validar dados do banco de dados
        $db_host = $_POST['db_host'] ?? 'localhost';
        $db_port = $_POST['db_port'] ?? '3306';
        $db_name = $_POST['db_name'] ?? 'financemanager';
        $db_user = $_POST['db_user'] ?? '';
        $db_password = $_POST['db_password'] ?? '';
        $app_url = $_POST['app_url'] ?? 'https://jtisuporte.com.br';
        
        if (empty($db_user)) {
            $errors[] = 'Usuário do banco de dados é obrigatório';
        }
        
        if (empty($errors)) {
            // Testar conexão
            $test = testDatabaseConnection($db_host, $db_user, $db_password);
            if (!$test['success']) {
                $errors[] = 'Erro ao conectar ao banco de dados: ' . $test['error'];
            } else {
                // Salvar dados na sessão
                $_SESSION['install_data'] = [
                    'db_host' => $db_host,
                    'db_port' => $db_port,
                    'db_name' => $db_name,
                    'db_user' => $db_user,
                    'db_password' => $db_password,
                    'app_url' => $app_url
                ];
                $step = 2;
            }
        }
    } elseif ($step == 2) {
        // Criar banco de dados
        $data = $_SESSION['install_data'];
        
        $createDb = createDatabase($data['db_host'], $data['db_user'], $data['db_password'], $data['db_name']);
        if (!$createDb['success']) {
            $errors[] = 'Erro ao criar banco de dados: ' . $createDb['error'];
        } else {
            $success[] = 'Banco de dados criado com sucesso';
            
            // Importar schema
            $import = importSchema($data['db_host'], $data['db_user'], $data['db_password'], $data['db_name']);
            if (!$import['success']) {
                $errors[] = 'Erro ao importar schema: ' . $import['error'];
            } else {
                $success[] = 'Schema importado com sucesso';
                
                // Salvar .env
                $saveEnv = saveEnvFile($data);
                if (!$saveEnv['success']) {
                    $errors[] = 'Erro ao salvar .env: ' . $saveEnv['error'];
                } else {
                    $success[] = 'Arquivo .env criado com sucesso';
                    
                    // Marcar como instalado
                    file_put_contents(__DIR__ . '/.installed', date('Y-m-d H:i:s'));
                    $step = 3;
                }
            }
        }
    }
}

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instalação - Finance Manager</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #2D5016 0%, #1A3A52 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .container {
            background: white;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            max-width: 600px;
            width: 100%;
            padding: 40px;
        }
        h1 {
            color: #2D5016;
            margin-bottom: 10px;
            font-size: 28px;
        }
        .subtitle {
            color: #666;
            margin-bottom: 30px;
            font-size: 14px;
        }
        .step-indicator {
            display: flex;
            justify-content: space-between;
            margin-bottom: 30px;
        }
        .step {
            width: 30%;
            height: 4px;
            background: #e0e0e0;
            border-radius: 2px;
            overflow: hidden;
        }
        .step.active {
            background: #2D5016;
        }
        .form-group {
            margin-bottom: 20px;
        }
        label {
            display: block;
            margin-bottom: 8px;
            color: #333;
            font-weight: 500;
            font-size: 14px;
        }
        input, select {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
            font-family: inherit;
        }
        input:focus, select:focus {
            outline: none;
            border-color: #2D5016;
            box-shadow: 0 0 0 3px rgba(45, 80, 22, 0.1);
        }
        .alert {
            padding: 12px 16px;
            border-radius: 6px;
            margin-bottom: 20px;
            font-size: 14px;
        }
        .alert-error {
            background: #fee;
            color: #c00;
            border: 1px solid #fcc;
        }
        .alert-success {
            background: #efe;
            color: #060;
            border: 1px solid #cfc;
        }
        .alert-info {
            background: #eef;
            color: #006;
            border: 1px solid #ccf;
        }
        .button-group {
            display: flex;
            gap: 10px;
            margin-top: 30px;
        }
        button {
            flex: 1;
            padding: 12px 20px;
            border: none;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }
        .btn-primary {
            background: #2D5016;
            color: white;
        }
        .btn-primary:hover {
            background: #1f3810;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(45, 80, 22, 0.3);
        }
        .btn-secondary {
            background: #e0e0e0;
            color: #333;
        }
        .btn-secondary:hover {
            background: #d0d0d0;
        }
        .info-box {
            background: #f5f5f5;
            padding: 16px;
            border-radius: 6px;
            margin-bottom: 20px;
            border-left: 4px solid #2D5016;
        }
        .info-box strong {
            display: block;
            margin-bottom: 8px;
            color: #2D5016;
        }
        .info-box p {
            font-size: 13px;
            color: #666;
            line-height: 1.5;
        }
        .success-message {
            text-align: center;
            padding: 40px 20px;
        }
        .success-icon {
            font-size: 48px;
            margin-bottom: 20px;
        }
        .success-message h2 {
            color: #2D5016;
            margin-bottom: 10px;
        }
        .success-message p {
            color: #666;
            margin-bottom: 20px;
        }
        .next-steps {
            background: #f5f5f5;
            padding: 20px;
            border-radius: 6px;
            margin-top: 20px;
        }
        .next-steps h3 {
            color: #2D5016;
            margin-bottom: 10px;
            font-size: 14px;
        }
        .next-steps ol {
            margin-left: 20px;
            font-size: 13px;
            color: #666;
            line-height: 1.8;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 Finance Manager</h1>
        <p class="subtitle">Instalação e Configuração</p>

        <div class="step-indicator">
            <div class="step <?php echo $step >= 1 ? 'active' : ''; ?>"></div>
            <div class="step <?php echo $step >= 2 ? 'active' : ''; ?>"></div>
            <div class="step <?php echo $step >= 3 ? 'active' : ''; ?>"></div>
        </div>

        <?php if ($step == 1): ?>
            <h2 style="color: #333; margin-bottom: 20px; font-size: 20px;">Passo 1: Configuração do Banco de Dados</h2>
            
            <div class="info-box">
                <strong>ℹ️ Informações Necessárias</strong>
                <p>Você precisará das credenciais do seu banco de dados MySQL. Geralmente encontradas no painel cPanel.</p>
            </div>

            <?php if (!empty($errors)): ?>
                <?php foreach ($errors as $error): ?>
                    <div class="alert alert-error">❌ <?php echo htmlspecialchars($error); ?></div>
                <?php endforeach; ?>
            <?php endif; ?>

            <form method="POST">
                <div class="form-group">
                    <label for="app_url">URL da Aplicação</label>
                    <input type="url" id="app_url" name="app_url" value="https://jtisuporte.com.br" required>
                    <small style="color: #999; display: block; margin-top: 4px;">Seu domínio com https://</small>
                </div>

                <div class="form-group">
                    <label for="db_host">Host do Banco de Dados</label>
                    <input type="text" id="db_host" name="db_host" value="localhost" required>
                    <small style="color: #999; display: block; margin-top: 4px;">Geralmente "localhost"</small>
                </div>

                <div class="form-group">
                    <label for="db_port">Porta do Banco de Dados</label>
                    <input type="text" id="db_port" name="db_port" value="3306" required>
                    <small style="color: #999; display: block; margin-top: 4px;">Geralmente "3306"</small>
                </div>

                <div class="form-group">
                    <label for="db_name">Nome do Banco de Dados</label>
                    <input type="text" id="db_name" name="db_name" value="financemanager" required>
                    <small style="color: #999; display: block; margin-top: 4px;">Nome que deseja para o banco</small>
                </div>

                <div class="form-group">
                    <label for="db_user">Usuário do Banco de Dados</label>
                    <input type="text" id="db_user" name="db_user" required>
                    <small style="color: #999; display: block; margin-top: 4px;">Usuário MySQL (encontre no cPanel)</small>
                </div>

                <div class="form-group">
                    <label for="db_password">Senha do Banco de Dados</label>
                    <input type="password" id="db_password" name="db_password">
                    <small style="color: #999; display: block; margin-top: 4px;">Deixe em branco se não houver senha</small>
                </div>

                <div class="button-group">
                    <button type="submit" class="btn-primary">Continuar →</button>
                </div>
            </form>

        <?php elseif ($step == 2): ?>
            <h2 style="color: #333; margin-bottom: 20px; font-size: 20px;">Passo 2: Criar Banco de Dados</h2>
            
            <div class="info-box">
                <strong>⚙️ Processando</strong>
                <p>Clique em "Instalar" para criar o banco de dados e importar as tabelas.</p>
            </div>

            <?php if (!empty($errors)): ?>
                <?php foreach ($errors as $error): ?>
                    <div class="alert alert-error">❌ <?php echo htmlspecialchars($error); ?></div>
                <?php endforeach; ?>
            <?php endif; ?>

            <?php if (!empty($success)): ?>
                <?php foreach ($success as $msg): ?>
                    <div class="alert alert-success">✅ <?php echo htmlspecialchars($msg); ?></div>
                <?php endforeach; ?>
            <?php endif; ?>

            <form method="POST">
                <input type="hidden" name="step" value="2">
                <div class="button-group">
                    <button type="submit" class="btn-primary">Instalar →</button>
                </div>
            </form>

        <?php elseif ($step == 3): ?>
            <div class="success-message">
                <div class="success-icon">✅</div>
                <h2>Instalação Concluída!</h2>
                <p>Finance Manager foi instalado com sucesso em sua hospedagem.</p>

                <div class="next-steps">
                    <h3>📋 Próximos Passos:</h3>
                    <ol>
                        <li><strong>Delete este arquivo</strong> (install.php) por segurança</li>
                        <li>Acesse <strong>https://jtisuporte.com.br</strong></li>
                        <li>Clique em <strong>"Registrar"</strong> para criar sua conta</li>
                        <li>Configure OAuth (Google/Apple) em <strong>.env</strong> (opcional)</li>
                        <li>Configure Open Banking em <strong>.env</strong> (opcional)</li>
                    </ol>
                </div>

                <div style="margin-top: 30px; padding: 20px; background: #fff3cd; border-radius: 6px; border-left: 4px solid #ffc107;">
                    <strong style="color: #856404;">⚠️ Importante:</strong>
                    <p style="color: #856404; margin-top: 8px; font-size: 13px;">
                        Após a instalação, delete o arquivo <strong>install.php</strong> da raiz do projeto por questões de segurança.
                    </p>
                </div>

                <div class="button-group" style="margin-top: 30px;">
                    <a href="https://jtisuporte.com.br" style="text-decoration: none;">
                        <button type="button" class="btn-primary" style="width: 100%;">Acessar Finance Manager →</button>
                    </a>
                </div>
            </div>

        <?php endif; ?>
    </div>
</body>
</html>
