# Documentação da API - Finance Manager

## Base URL

```
https://seu-dominio.com/api
```

## Autenticação

Todas as requisições devem incluir o header de autenticação:

```
Authorization: Bearer {token}
```

O token é obtido após login/registro bem-sucedido.

## Endpoints

### Autenticação

#### Registrar
```
POST /auth/register
Content-Type: application/json

{
  "name": "João Silva",
  "email": "joao@example.com",
  "password": "senha123",
  "password_confirm": "senha123"
}

Response:
{
  "success": true,
  "token": "eyJ0eXAiOiJKV1QiLCJhbGc...",
  "user": {
    "id": 1,
    "name": "João Silva",
    "email": "joao@example.com"
  }
}
```

#### Login
```
POST /auth/login
Content-Type: application/json

{
  "email": "joao@example.com",
  "password": "senha123"
}

Response:
{
  "success": true,
  "token": "eyJ0eXAiOiJKV1QiLCJhbGc...",
  "user": {
    "id": 1,
    "name": "João Silva",
    "email": "joao@example.com"
  }
}
```

#### Logout
```
POST /auth/logout
Authorization: Bearer {token}

Response:
{
  "success": true,
  "message": "Logout realizado com sucesso"
}
```

### Transações

#### Listar Transações
```
GET /transactions/list?page=1&limit=20&type=&account_id=&category_id=
Authorization: Bearer {token}

Response:
{
  "success": true,
  "transactions": [
    {
      "id": 1,
      "user_id": 1,
      "account_id": 1,
      "category_id": 1,
      "type": "expense",
      "amount": 50.00,
      "description": "Almoço",
      "transaction_date": "2024-01-15",
      "status": "completed",
      "created_at": "2024-01-15 12:30:00"
    }
  ]
}
```

#### Criar Transação
```
POST /transactions/create
Authorization: Bearer {token}
Content-Type: application/json

{
  "account_id": 1,
  "category_id": 1,
  "type": "expense",
  "amount": 50.00,
  "description": "Almoço",
  "transaction_date": "2024-01-15"
}

Response:
{
  "success": true,
  "message": "Transação criada com sucesso",
  "transaction_id": 1
}
```

#### Atualizar Transação
```
PUT /transactions/update?id=1
Authorization: Bearer {token}
Content-Type: application/json

{
  "amount": 55.00,
  "description": "Almoço e café"
}

Response:
{
  "success": true,
  "message": "Transação atualizada com sucesso"
}
```

#### Deletar Transação
```
DELETE /transactions/delete?id=1
Authorization: Bearer {token}

Response:
{
  "success": true,
  "message": "Transação deletada com sucesso"
}
```

### Contas

#### Listar Contas
```
GET /accounts/list
Authorization: Bearer {token}

Response:
{
  "success": true,
  "accounts": [
    {
      "id": 1,
      "user_id": 1,
      "name": "Conta Corrente",
      "account_type": "checking",
      "bank_name": "Banco do Brasil",
      "balance": 1500.00,
      "color": "#2D5016",
      "is_active": true,
      "created_at": "2024-01-01"
    }
  ]
}
```

#### Criar Conta
```
POST /accounts/create
Authorization: Bearer {token}
Content-Type: application/json

{
  "name": "Conta Poupança",
  "account_type": "savings",
  "bank_name": "Banco do Brasil",
  "balance": 0,
  "color": "#2D5016"
}

Response:
{
  "success": true,
  "message": "Conta criada com sucesso",
  "account_id": 2
}
```

### Categorias

#### Listar Categorias
```
GET /categories/list?type=
Authorization: Bearer {token}

Response:
{
  "success": true,
  "categories": [
    {
      "id": 1,
      "user_id": 1,
      "name": "Alimentação",
      "type": "expense",
      "color": "#FFA500",
      "is_active": true
    }
  ]
}
```

#### Criar Categoria
```
POST /categories/create
Authorization: Bearer {token}
Content-Type: application/json

{
  "name": "Transporte",
  "type": "expense",
  "color": "#FF6B6B"
}

Response:
{
  "success": true,
  "message": "Categoria criada com sucesso",
  "category_id": 2
}
```

### Tags

#### Listar Tags
```
GET /tags/list?hierarchy=true
Authorization: Bearer {token}

Response:
{
  "success": true,
  "tags": [
    {
      "id": 1,
      "user_id": 1,
      "name": "Viagem",
      "color": "#FFA500",
      "subtags": [
        {
          "id": 2,
          "name": "Passagem Aérea",
          "color": "#FF8C00"
        }
      ]
    }
  ]
}
```

#### Criar Tag
```
POST /tags/create
Authorization: Bearer {token}
Content-Type: application/json

{
  "name": "Projeto X",
  "parent_tag_id": null,
  "color": "#FFA500"
}

Response:
{
  "success": true,
  "message": "Tag criada com sucesso",
  "tag_id": 1
}
```

#### Adicionar Tag a Transação
```
POST /tags/add-to-transaction
Authorization: Bearer {token}
Content-Type: application/json

{
  "transaction_id": 1,
  "tag_id": 1
}

Response:
{
  "success": true,
  "message": "Tag adicionada à transação"
}
```

### Relatórios

#### Obter Relatório
```
GET /reports/get?start_date=2024-01-01&end_date=2024-01-31&type=&account_id=&category_id=
Authorization: Bearer {token}

Response:
{
  "success": true,
  "transactions": [...],
  "totals": {
    "total_income": 5000.00,
    "total_expense": 2500.00,
    "total_transactions": 50
  },
  "summary_by_category": [...],
  "summary_by_account": [...]
}
```

#### Resumo por Período
```
GET /reports/summary-by-period?start_date=2024-01-01&end_date=2024-01-31&group_by=day
Authorization: Bearer {token}

Response:
{
  "success": true,
  "summary": [
    {
      "period": "2024-01-01",
      "type": "income",
      "count": 5,
      "total": 1000.00
    }
  ]
}
```

#### Resumo por Categoria
```
GET /reports/summary-by-category?start_date=2024-01-01&end_date=2024-01-31&type=expense
Authorization: Bearer {token}

Response:
{
  "success": true,
  "summary": [
    {
      "id": 1,
      "name": "Alimentação",
      "color": "#FFA500",
      "count": 20,
      "total": 500.00
    }
  ]
}
```

#### Exportar PDF
```
GET /reports/export-pdf?start_date=2024-01-01&end_date=2024-01-31
Authorization: Bearer {token}

Response: PDF file download
```

### Open Banking

#### Listar Conexões
```
GET /open-banking/connections
Authorization: Bearer {token}

Response:
{
  "success": true,
  "connections": [
    {
      "id": 1,
      "user_id": 1,
      "bank_code": "001",
      "institution_name": "Banco do Brasil",
      "account_number": "12345-6",
      "is_active": true,
      "last_sync_at": "2024-01-15 10:30:00"
    }
  ]
}
```

#### Obter URL de Autorização
```
GET /open-banking/auth-url?bank_code=001
Authorization: Bearer {token}

Response:
{
  "success": true,
  "authorization_url": "https://api.openbanking.com/authorize?..."
}
```

#### Sincronizar Transações
```
POST /open-banking/sync?connection_id=1
Authorization: Bearer {token}
Content-Type: application/json

{
  "start_date": "2024-01-01",
  "end_date": "2024-01-31"
}

Response:
{
  "success": true,
  "message": "Sincronização concluída",
  "imported": 25,
  "total": 25
}
```

## Códigos de Status HTTP

| Código | Significado |
|--------|-------------|
| 200 | OK - Requisição bem-sucedida |
| 201 | Created - Recurso criado com sucesso |
| 400 | Bad Request - Dados inválidos |
| 401 | Unauthorized - Não autenticado |
| 403 | Forbidden - Sem permissão |
| 404 | Not Found - Recurso não encontrado |
| 405 | Method Not Allowed - Método HTTP não permitido |
| 500 | Internal Server Error - Erro no servidor |

## Tratamento de Erros

Todas as respostas de erro seguem este formato:

```json
{
  "error": "Descrição do erro",
  "code": "ERROR_CODE"
}
```

## Rate Limiting

- Limite: 1000 requisições por hora
- Header: `X-RateLimit-Remaining`

## Versionamento

A API utiliza versionamento por URL:

```
https://seu-dominio.com/api/v1/...
```

## Exemplos de Uso

### cURL

```bash
# Login
curl -X POST https://seu-dominio.com/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{"email":"joao@example.com","password":"senha123"}'

# Listar transações
curl -X GET https://seu-dominio.com/api/transactions/list \
  -H "Authorization: Bearer {token}"
```

### JavaScript

```javascript
// Login
const response = await fetch('https://seu-dominio.com/api/auth/login', {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify({
    email: 'joao@example.com',
    password: 'senha123'
  })
});

const data = await response.json();
const token = data.token;

// Listar transações
const transResponse = await fetch('https://seu-dominio.com/api/transactions/list', {
  headers: { 'Authorization': `Bearer ${token}` }
});
```

### Python

```python
import requests

# Login
response = requests.post('https://seu-dominio.com/api/auth/login', json={
    'email': 'joao@example.com',
    'password': 'senha123'
})

token = response.json()['token']

# Listar transações
headers = {'Authorization': f'Bearer {token}'}
response = requests.get('https://seu-dominio.com/api/transactions/list', headers=headers)
```
